<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Model\Product;

use Includes\Utils\ArrayManager;
use XCart\Extender\Mapping\Extender;
use XLite\Model\Cart;
use XLite\Model\Product;

/**
 * @Extender\Mixin
 */
abstract class ProductStockAvailabilityPolicy extends \XLite\Model\Product\ProductStockAvailabilityPolicy
{
    public const PRODUCT_IS_AVAILABLE_FOR_BACKORDER = 'backorder_available';
    public const IS_BACKORDER_LIMIT                 = 'backorder_limited';
    public const BACKORDER_LIMIT                    = 'backorder_limit';
    public const MAX_QUANTITY_ALLOWED               = 'max_quantity_allowed';

    public function getAvailableBackorderAmount(Cart $cart)
    {
        $return = 0;

        if ($this->dto[static::IS_BACKORDER_LIMIT]) {
            // $cartItems  = $cart->getItemsByProductId($this->dto[static::PRODUCT_ID]);
            // $cartBackorderAmount = ArrayManager::sumObjectsArrayFieldValues($cartItems, 'getBackorderAmount', true);
            $cartBackorderAmount = $this->getInCartBackorderAmount($cart);

            $return = max($this->dto[static::BACKORDER_LIMIT] - $cartBackorderAmount, 0);
        }

        return $return;
    }

    public function getInCartBackorderAmount(Cart $cart)
    {
        $cartItems  = $cart->getItemsByProductId($this->dto[static::PRODUCT_ID]);

        return ArrayManager::sumObjectsArrayFieldValues($cartItems, 'getBackorderAmount', true);
    }

    /**
     * Create a data transfer object out of the Product instance.
     * This object should contain all of the data necessary for getAvailableAmount() & canAddToCart() methods to compute their value.
     *
     * @param Product $product
     *
     * @return array
     */
    protected function createDTO(Product $product)
    {
        $return = parent::createDTO($product);

        $return += [
            static::PRODUCT_IS_AVAILABLE_FOR_BACKORDER
                                          => $product->getIsAvailableForBackorder(),
            static::IS_BACKORDER_LIMIT    => $product->getIsBackorderLimit(),
            static::BACKORDER_LIMIT       => $product->getBackorderLimit(),
            static::MAX_QUANTITY_ALLOWED  => $product->getMaxPurchaseLimit()
        ];

        return $return;
    }

    public function getAvailableAmountIncludingBackorder(Cart $cart)
    {
        $result = parent::getAvailableAmount($cart);

        if (!empty($this->dto[static::PRODUCT_IS_AVAILABLE_FOR_BACKORDER])) {
            if (empty($this->dto[static::IS_BACKORDER_LIMIT])) {
                $result = $this->dto[static::MAX_QUANTITY_ALLOWED];
            } else {
                $result += (int)($this->dto[static::BACKORDER_LIMIT] ?? 0);
            }
        }

        return min(
            ($this->dto[static::MAX_QUANTITY_ALLOWED] ?? 0),
            $result
        );
    }

    /**
     * @return boolean
     */
    public function isAllStockInCart(Cart $cart)
    {
        return $this->getAvailableAmountIncludingBackorder($cart) <= $this->getInCartAmount($cart);
    }
}
