<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Model;

use Doctrine\ORM\Mapping as ORM;
use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 * @Extender\After ("XC\ProductVariants")
 */
abstract class Product extends \XLite\Model\Product
{
    /**
     * @ORM\Column (type="boolean", nullable=true)
     */
    protected $isAvailableForBackorder = false;

    /**
     * @ORM\Column (type="boolean", nullable=true)
     */
    protected $isBackorderLimit = false;

    /**
     * @ORM\Column (type="integer", nullable=true)
     */
    protected $backorderLimit = 0;


    protected $totalBackorderAmount = null;

    protected $totalBackorderClosedAmount = null;

    public function getAvailableBackorderAmount()
    {
        return $this->getStockAvailabilityPolicy()->getAvailableBackorderAmount(\XLite\Model\Cart::getInstance());
    }

    /**
     * Number of items to be backordered currently in user's cart
     *
     * @return int
     **/
    public function getInCartBackorderAmount()
    {
        return $this->getStockAvailabilityPolicy()->getInCartBackorderAmount(\XLite\Model\Cart::getInstance());
    }

    /**
     * How many product items added to cart
     *
     * @return boolean
     */
    public function getItemsInCartMessage()
    {
        $res = parent::getItemsInCartMessage();

        $cartBackorderAmount = $this->getInCartBackorderAmount();
        if ($cartBackorderAmount) {
            $res .= '<br><i class="icon-pencil"></i> ' . self::t(
                'X of them will be backordered',
                [
                    'backordered' => $cartBackorderAmount,
                ]
            );
        }

        return $res;
    }

    public function getRemainingBackorderAmount()
    {
        $res = $this->getBackorderLimit() - $this->calculateTotalBackorderAmount() + $this->calculateTotalBackorderClosedAmount();
        return "{$this->getBackorderLimit()} - {$this->calculateTotalBackorderAmount()} + {$this->calculateTotalBackorderClosedAmount()} = {$res}";
    }

    public function calculateTotalBackorderAmount()
    {
        if ($this->totalBackorderAmount === null) {
            $this->totalBackorderAmount = (int)\XLite\Core\Database::getEM()->createQueryBuilder()
                ->select('SUM(oi.backorderAmount)')
                ->from(\XLite\Model\OrderItem::class, 'oi')
                ->where('oi.object = :product')
                ->andWhere('oi.backorderAmount > 0')
                ->setParameter('product', $this)
                ->getQuery()->getSingleScalarResult();
        }

        return $this->totalBackorderAmount;
    }

    public function calculateTotalBackorderClosedAmount()
    {
        if ($this->totalBackorderClosedAmount === null) {
            $this->totalBackorderClosedAmount = (int)\XLite\Core\Database::getEM()->createQueryBuilder()
                ->select('SUM(oi.backorderClosedAmount)')
                ->from(\XLite\Model\OrderItem::class, 'oi')
                ->where('oi.object = :product')
                ->andWhere('oi.backorderClosedAmount > 0')
                ->setParameter('product', $this)
                ->getQuery()->getSingleScalarResult();
        }

        return $this->totalBackorderClosedAmount;
    }

    public function changeBackorderLimit($delta)
    {
        $this->setBackorderLimit($this->getBackorderLimit() + $delta);
        return $this;
    }

    /**
     * @return boolean
     */
    public function canBeBackordered()
    {
        return ($this->getInventoryEnabled() || !parent::availableInDate())
            && ($this->isUnlimitedBackorder() || $this->isLimitedBackorder());
    }

    /**
     * for customer zone
     *
     * @return boolean
     */
    public function canBePreordered()
    {
        return $this->canBeBackordered()
            && (!$this->getInventoryEnabled() || ($this->getInventoryEnabled() && $this->getPublicAmount() <= 0))
            && !parent::availableInDate();
    }

    public function isUnlimitedBackorder()
    {
        return $this->getIsAvailableForBackorder() && !$this->getIsBackorderLimit();
    }

    public function isLimitedBackorder()
    {
        return $this->getIsAvailableForBackorder()
            && $this->getIsBackorderLimit()
            && $this->getBackorderLimit() > 0;
    }

    /**
     * @return boolean
     */
    public function availableInDate()
    {
        return $this->canBeBackordered() ?: parent::availableInDate();
    }

    /**
     * @return boolean
     */
    public function availableInDateOrigin()
    {
        return parent::availableInDate();
    }

    public function isOutOfStock()
    {
        return parent::isOutOfStock()
            && !$this->canBeBackordered();
    }

    /**
     * Get default variant
     *
     * @return mixed
     */
    public function getDefaultVariant()
    {
        if ($this->hasVariants()) {
            $this->defaultVariant = parent::getDefaultVariant();

            // In cusomer area we double-check if the item is "trully" in-stock.
            // If not, we try to assign the one that is trully.
            //
            // *trully = not just available for purchase (like out of stock, but available for backorder)
            // , but is present in stock right now
            if (! \XLite::isAdminZone() && $this->defaultVariant && $this->defaultVariant->isOutOfStockDefault()) {
                $this->setFirstAvailableVariantAsDefault();
            }
        }

        return $this->defaultVariant ?: null;
    }

    /**
     * Set first available variant as default
     *
     * @return void
     */
    public function setFirstAvailableVariantAsDefault()
    {
        foreach ($this->getVariants() as $variant) {
            if (! $variant->isOutOfStockDefault()) {
                $this->defaultVariant = $variant;
                return;
            }
        }
    }

    public function getLowAvailableAmount()
    {
        return $this->getIsAvailableForBackorder()
            ? max(parent::getLowAvailableAmount(), $this->getLowDefaultAmount())
            : parent::getLowAvailableAmount();
    }

    public function getIsAvailableForBackorder()
    {
        return $this->isAvailableForBackorder;
    }

    public function setIsAvailableForBackorder($value)
    {
        $this->isAvailableForBackorder = $value;
        return $this;
    }

    public function getIsBackorderLimit()
    {
        return $this->isBackorderLimit;
    }

    public function setIsBackorderLimit($flag)
    {
        $this->isBackorderLimit = $flag;
        return $this;
    }

    public function getBackorderLimit()
    {
        return $this->backorderLimit;
    }

    public function setBackorderLimit($amount)
    {
        $this->backorderLimit = $amount;
        return $this;
    }

    /**
     * Alias: is all product items in cart
     *
     * @return boolean
     */
    public function isAllStockInCart()
    {
        $res = parent::isAllStockInCart();
        if (! $res) {
            return false;
        }

        if ($this->isUnlimitedBackorder()) {
            return $this->getItemsInCart() >= $this->getMaxPurchaseLimit();
        } elseif (
            $this->isLimitedBackorder()
            && $this->getAvailableBackorderAmount() > 0
        ) {
            return false;
        }

        return true;
    }

    /**
     * Check if upcoming product is available
     *
     * @return boolean
     */
    public function isAllowedUpcomingProduct()
    {
        return $this->getIsAvailableForBackorder();
    }

    /**
     * @return string
     */
    public function getBackorderQtyLabel()
    {
        return $this->getTranslationField(__FUNCTION__);
    }

    /**
     * @param $backorderQtyLabel
     *
     * @return \XLite\Model\Base\Translation
     */
    public function setBackorderQtyLabel($backorderQtyLabel)
    {
        return $this->setTranslationField(__FUNCTION__, $backorderQtyLabel);
    }

    public function getAvailableAmountForTooltipText()
    {
        $result = parent::getAvailableAmountForTooltipText();

        if ($this->canBeBackordered()) {
            if ($this->isLimitedBackorder()) {
                $result += $this->getBackorderLimit();
            } else {
                $result = $this->getMaxPurchaseLimit();
            }
        }

        return min($this->getMaxPurchaseLimit(), $result);
    }
}
