<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Model;

use Doctrine\ORM\Mapping as ORM;
use Includes\Utils\ArrayManager;
use XCart\Extender\Mapping\Extender;

/**
 * Decorated OrderItem model.
 *
 * Note: this has to be after ProductVariants due to getProductAvailableAmount().
 *
 * @Extender\Mixin
 * @Extender\After ("XC\ProductVariants")
 */
abstract class OrderItem extends \XLite\Model\OrderItem
{
    /**
     * Number of backordered units.
     *
     * @ORM\Column (type="integer")
     */
    protected $backorderAmount = 0;

    /**
     * Number of backordered units that have been received from the supplier and
     * shipped to the customer.
     *
     * @ORM\Column (type="integer")
     */
    protected $backorderClosedAmount = 0;


    protected $previousBackorderAmount = 0;

    public $wasRegistered = false;

    /**
     * Modified setter
     *
     * @param integer $amount Value to set
     *
     * @return void
     */
    public function setAmount($amount)
    {
        $doNotUpdate = false;

        // I'm not sure why there is this check (added by previous developer),
        // but without the extra isPersistent() rule this code caused
        // backordered items with qty=1 to disappear from backordering stats
        if ($this->isPersistent() && ($amount == $this->getAmount())) {
            $doNotUpdate = true;
        }

        parent::setAmount($amount);

        if (! $doNotUpdate && $this->getProduct()->canBeBackordered()) {
            $this->renewBackorderAmount();
        }
    }

    /**
     * Updates the number of backordered units depending on the current
     * product stock and backorder settings.
     *
     * @return void
     */
    protected function renewBackorderAmount()
    {
        $this->setPreviousBackorderAmount(
            $this->isNewOrder() ? 0 : $this->getBackorderAmount()
        );

        $backorderQty = $this->getProduct()->canBeBackordered()
            ? ($this->getAmount() - $this->getItemPublicAmount())
            : 0;

        $this->setBackorderAmount(($backorderQty > 0) ? $backorderQty : 0);
    }

    public function setPreviousBackorderAmount($amount)
    {
        $this->previousBackorderAmount = $amount;
    }

    /**
     * Returns the number of backordered units.
     *
     * @return int
     */
    public function getBackorderAmount()
    {
        return $this->backorderAmount;
    }

    /**
     * Updates the number of backordered units.
     *
     * @param int $amount New number of backordered units.
     *
     * @return $this
     */
    public function setBackorderAmount($amount)
    {
        $this->backorderAmount = $amount;

        return $this;
    }

    /**
     * Returns the number of backordered units that have been received
     * from the supplier and shipped to the customer.
     *
     * @return int
     */
    public function getBackorderClosedAmount()
    {
        return $this->backorderClosedAmount;
    }

    /**
     * Updates the number of backordered units that have been received
     * from the supplier and shipped to the customer.
     *
     * @param int $amount New number of units.
     *
     * @return $this
     */
    public function setBackorderClosedAmount($amount)
    {
        $this->backorderClosedAmount = $amount;

        return $this;
    }

    public function getPreviousBackorderAmount()
    {
        return (int) $this->previousBackorderAmount;
    }

    /**
     * Check if item has a wrong amount
     *
     * @return boolean
     */
    public function hasWrongAmount()
    {
        // TODO: refactor when #0049295 is solved - at the moment we completely
        // ignore the parent::hasWrongAmount() result for backordered products.
        // Because of this we have to duplicate the checks added by other
        // modules to hasWrongAmount() in our custom hasWrongBackorderAmount()
        // method too, but this may cause some checks from unknown modules
        // being ignored. Refactor this dirty hack when possible.
        return $this->isAvailableForBackorder()
            ? $this->hasWrongBackorderAmount()
            : parent::hasWrongAmount();
    }

    /**
     * Get available amount for the product
     *
     * @return integer
     */
    public function getProductAvailableAmount()
    {
        $product = $this->getProduct();

        if ($product->isUnlimitedBackorder()) {
            return $product->getMaxPurchaseLimit() - $this->getAmount();
        }

        return ($product->isLimitedBackorder() ? $product->getBackorderLimit() : 0)
            + parent::getProductAvailableAmount();
    }

    /**
     * Get available amount for the product (as if Backorder hadn't been enabled).
     *
     * @return integer
     */
    public function getProductAvailableAmountWithoutBackorder()
    {
        return $this->canBePreordered() ? 0 : parent::getProductAvailableAmount();
    }


    /**
     * Renew order item
     *
     * @return boolean
     */
    public function renew()
    {
        $available = parent::renew();

        if ($available) {
            $this->renewBackorderAmount();
        }

        return $available;
    }

    /**
     * Check - can change item's amount or not
     *
     * @return boolean
     */
    public function canChangeAmount()
    {
        return parent::canChangeAmount()
            || ($this->isAvailableForBackorder() && $this->canChangeBackorderAmount());
    }

    /**
     * Checks if the item can be backordered.
     *
     * @return boolean
     */
    protected function isAvailableForBackorder()
    {
        return $this->getProduct()->getIsAvailableForBackorder();
    }

    /**
     * Check whether it is possible to change the amount for a backordered item.
     *
     * @return boolean
     */
    protected function canChangeBackorderAmount()
    {
        return true;
        $product = $this->getProduct();

        return ! $product->getIsBackorderLimit()
            || (0 < ($this->getItemPublicAmount() + $this->getItemBackorderLimit()));
    }

    /**
     * Checks if the number of added/backordered units is wrong.
     *
     * TODO: refactor when #0049295 is solved
     *
     * @return boolean
     */
    protected function hasWrongBackorderAmount()
    {
        $result = $this->hasBackorderedMoreThanInStock();

        if (method_exists($this, 'hasWrongMinQuantity')) {
            // For some reasons moving this into a separate class that depends
            // on Wholesale module results into either a circullar dependency
            // or the decoration being happened before this method. That's why
            // we have to check for Wholesale module in this tricky/dirty way.
            $result = $result || $this->hasWrongMinQuantity();
        }

        return $result;
    }

    /**
     * Checks if the user added more units that can be bought or backordered.
     *
     * @return boolean
     */
    protected function hasBackorderedMoreThanInStock()
    {
        $product = $this->getProduct();

        return $product->getInventoryEnabled()
            && $product->getIsBackorderLimit()
            && ($this->getAmount() > ($this->getItemPublicAmount() + $this->getItemBackorderLimit()));
    }

    /**
     * Returns the public product amount for the item.
     *
     * @return int
     */
    protected function getItemPublicAmount()
    {
        return $this->getProduct()->getPublicAmount();
    }

    /**
     * Returns the maximum number of units that can be backordered for the item.
     *
     * @return int
     */
    protected function getItemBackorderLimit()
    {
        if (! $this->isNewOrder()) {
            return max(0, $this->getProduct()->getBackorderLimit());
        }

        return max(
            0,
            $this->getProduct()->getBackorderLimit()
                - $this->getOtherPositionsCount()
        );
    }

    protected function getOtherPositionsCount()
    {
        $cartItems = $this->getOrder()->getItemsByProductId($this->getProduct()->getProductId());
        $allItemsAmount = ArrayManager::sumObjectsArrayFieldValues($cartItems, 'getBackorderAmount', true);

        return $allItemsAmount - $this->getBackorderAmount();
    }

    /**
     * @return boolean
     */
    public function canBePreordered()
    {
        $product = $this->getProduct();
        $availableAmount = $this->getAmount() - $this->getBackorderAmount();
        $amount = $availableAmount ? $this->getItemPublicAmount() - $availableAmount : $this->getItemPublicAmount();

        return ($product->canBePreordered() || ($this->getBackorderAmount() > 0 && $amount <= 0))
            && !$product->availableInDateOrigin();
    }

    /**
     * Increase / decrease product inventory amount
     *
     * @param integer $delta Amount delta
     *
     * @return void
     */
    public function changeAmount($delta)
    {
        $itemQty = $newItemQty = $this->getInventoryAmount();

        $backorderAmount = $newBackorderAmount = $this->previousBackorderAmount;

        $absDelta = abs($delta);

        // Special protocol for backordered items (when changing ordered item's qty)
        //
        // 1. Delta is positive (returning to stock)
        //    (in this case, might be possible to ship this order right way, so we nullify backorderAmount firsthand)
        //    - split delta on two summands: 1. up until order item's backorderAmount and 2. the remaining
        //    - 1st summand decreases order item's backorderAmount and increases product's backorderLimit
        //    - 2nd summand increases product's stock
        //
        //    Example: there's an order item record, where amount=10 and backorderAmount=5
        //      - admin changes item amount to 3, which is delta=-7 (so: 1stSummand=5, 2ndSummand=2)
        //      - step 1: change order item's backorderAmount to 5 - 5 = 0
        //      - step 2: increase product's backorderLimit by 5
        //      - step 3: increase product's QtyInStock by 2
        if ($delta > 0) {
            // ... split delta on two summands: 1. up until order item's backorderAmount and 2. the remaining
            $returnToBackstockPart = min($absDelta, $backorderAmount); // ignore backorderClosedAmount until after appropriate test cases are ready…
            $returnToStockPart = max($absDelta - $returnToBackstockPart, 0);

            if ($returnToBackstockPart > 0) {
                // ... 1st summand decreases order item's backorderAmount
                $newBackorderAmount = $backorderAmount - $returnToBackstockPart;
                $this->setBackorderAmount($newBackorderAmount);

                // ... and increases product's backorderLimit
                if ($this->getProduct()->getIsBackorderLimit()) {
                    $this->getProduct()->setBackorderLimit(
                        $this->getProduct()->getBackorderLimit() + $returnToBackstockPart
                    );
                }
            }

            // ... 2nd summand increases product's stock
            if ($returnToStockPart > 0) {
                $newItemQty = $itemQty + $returnToStockPart;
                parent::changeAmount($returnToStockPart);
            }
        }

        // 2. Delta is negative (taking more from the stock)
        //    - split delta on two summands: 1. up until product's current Qty in stock and 2. the remaining
        //    - 1st summand decreases current product quantity (until depletion)
        //    - 2nd summand decreases product's backorderLimit and increases order item's backorderAmount
        //
        //    Example: there's an order item record, where amount=10 and backorderAmount=5
        //             product's current QtyInStock is 2, backorderLimit — let say "10" (just for example)
        //      - admin changes item amount to 15, which is delta=+5 (so: 1stSummand=2, 2ndSummand=3)
        //      - step 1: deplete product's QtyInStock: 2 -> 0
        //      - step 2: decrease product's backorderLimit by 3: 10 -> 7
        //      - step 3: increases order item's backorderAmount by 3: 5 -> 8
        //
        //      Before:
        //          - ordered 10 items, 5 of them are backordered
        //          - the product has 2 items available immediately and 10 more are available for backorder
        //
        //      After:
        //          - ordered 15 items, 8 of them backordered
        //          - the product has 0 items available immediately and 7 more are available for backorder
        if ($delta < 0) {
            // ... split delta on two summands: 1. up until product's current Qty in stock and 2. the remaining
            if ($this->getProduct()->canBeBackordered()) {
                if ($this->getProduct()->canBePreordered()) {
                    $itemQty = 0;
                }

                $takeFromStockPart = min($absDelta, $itemQty);
                $takeFromBackstockPart = max($absDelta - $takeFromStockPart, 0);
            } else {
                $takeFromStockPart = $absDelta;
                $takeFromBackstockPart = 0;

                if ($absDelta > $itemQty) {
                    \XLite\Core\OrderHistory::getInstance()->registerEvent(
                        $this->getOrder()->getOrderId(),
                        \XLite\Core\OrderHistory::CODE_CHANGE_AMOUNT,
                        \XLite\Core\OrderHistory::TXT_UNABLE_RESERVE_AMOUNT,
                        [
                            'product' => $this->getExtendedItemName(),
                            'qty' => $takeFromStockPart,
                        ]
                    );

                    return;
                }
            }

            if ($takeFromStockPart > 0) {
                // ... 1st summand decreases current product quantity (until depletion)
                $newItemQty = $itemQty - $takeFromStockPart;
                parent::changeAmount(-1 * $takeFromStockPart);
            }

            if ($takeFromBackstockPart > 0) {
                // ... 2nd summand decreases product's backorderLimit
                if ($this->getProduct()->getIsBackorderLimit()) {
                    $newLimit = max(0, $this->getProduct()->getBackorderLimit() - $takeFromBackstockPart);
                    $this->getProduct()->setBackorderLimit($newLimit);
                }

                // ... and increases order item's backorderAmount
                $newBackorderAmount = $backorderAmount + $takeFromBackstockPart;
                $this->setBackorderAmount($newBackorderAmount);
            }
        }

        // Register changes in the order history:
        // - always, when it's not new order placed
        // - if it's a new order, but inventory notifications are not groupped
        //   (there's separate procedure for groupped notifications)
        if (! $this->isNewOrder() || ! $this->isGroupData()) {
            \XLite\Core\OrderHistory::getInstance()->registerChangeAmountForBackordered(
                $this->getOrder()->getOrderId(),
                $this->getProduct(),
                $this,
                $itemQty,
                $newItemQty,
                $backorderAmount,
                $newBackorderAmount
            );
        }
    }

    protected function isNewOrder()
    {
        return ! \XLite::isAdminZone();
    }

    // Inventory notifications are groupped and processed separately
    protected function isGroupData()
    {
        return 1 < count($this->getOrder()->getItems());
    }

    public function getExtendedItemName()
    {
        return trim("{$this->getName()} {$this->getExtendedDescription()}");
    }
}
