<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Model;

use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 */
class Order extends \XLite\Model\Order
{
    /**
     * Return true if item amount limit is reached
     *
     * @param \XLite\Model\OrderItem $item Order item
     *
     * @return boolean
     */
    protected function isItemLimitReached($item)
    {
        if (parent::isItemLimitReached($item)) {
            /** @var \XLite\Model\Product $product */
            $product = $item->getProduct();

            if (!$product->getIsAvailableForBackorder()) {
                return true;
            }

            $maxPurchaseLimit = $product->getMaxPurchaseLimit();
            $itemAmount = $item->getAmount();

            if ($product->getIsBackorderLimit()) {
                $maxAllowedLimit = $item->getObject()->getPublicAmount() + $product->getBackorderLimit();

                if (min($maxPurchaseLimit, $maxAllowedLimit) <= $itemAmount) {
                    return true;
                }
            } elseif ($itemAmount >= $maxPurchaseLimit) {
                return true;
            }
        }

        return false;
    }

    /**
     * Called when an order successfully placed by a client
     *
     * @return void
     */
    public function processSucceed()
    {
        $this->beforeProcessSucceedBackorder();
        parent::processSucceed();
        $this->afterProcessSucceedBackorder();
    }

    protected function processBackorderedItems()
    {
        return $this->isBackorder() ? null : parent::processBackorderedItems();
    }

    /**
     * Implements the actions to perform before calling the processSucceed().
     *
     * @return void
     */
    protected function beforeProcessSucceedBackorder()
    {
        if ($this->isPreorder()) {
            $this->setShippingStatus(\XLite\Model\Order\Status\Shipping::STATUS_PRE_ORDERED);
        } elseif ($this->isBackorder()) {
            $this->setShippingStatus(\XLite\Model\Order\Status\Shipping::STATUS_BACK_ORDERED);
        }
    }

    /**
     * Implements the actions to perform after calling the processSucceed().
     *
     * @return void
     */
    protected function afterProcessSucceedBackorder()
    {
        $this->processOpenBackorder();
    }

    /**
     * Checks if the order has backordered items.
     *
     * @return boolean
     */
    protected function isBackorder()
    {
        foreach ($this->getItems() as $item) {
            if ($item->getBackorderAmount() > 0) {
                return true;
            }
        }

        return false;
    }

    protected function isPreorder()
    {
        foreach ($this->getItems() as $item) {
            if ($item->canBePreordered()) {
                return true;
            }
        }

        return false;
    }

    /**
     * Marks backordered items as received from suppliers.
     *
     * Implements the logic for the "sendBackorder" handler for
     * the shipping status change event.
     *
     * @return void
     */
    protected function processSendBackorder()
    {
        foreach ($this->getItems() as $item) {
            if (($bAmount = $item->getBackorderAmount()) > 0) {
                $item->setBackorderClosedAmount($bAmount);
            }
        }
    }

    /**
     * Marks backordered items as not received from suppliers.
     *
     * Implements the logic for the "openBackorder" handler for
     * the shipping status change event.
     *
     * @return void
     */
    protected function processOpenBackorder()
    {
        foreach ($this->getItems() as $item) {
            if ($item->getBackorderClosedAmount() > 0) {
                $item->setBackorderClosedAmount(0);
            }
        }
    }

    /**
     * Increase / decrease item product inventory
     *
     * @param \XLite\Model\OrderItem $item      Order item
     * @param integer                $sign      Flag; "1" or "-1"
     * @param boolean                $register  Register in order history OPTIONAL
     *
     * @return integer
     */
    protected function changeItemInventory($item, $sign, $register = true)
    {
        $amount = $sign * $item->getAmount();

        if ($register && ! $this->isOrderingAvailableForBackorderProduct($sign, $item)) {
            $this->registerHistoryChangeItemAmount($item, $amount);
        }

        if ($sign > 0) { // returning back to stock
            $item->setPreviousBackorderAmount($item->getBackorderAmount());
        }

        $item->changeAmount($amount);

        return $amount;
    }

    protected function isOrderingAvailableForBackorderProduct($sign, \XLite\Model\OrderItem $item)
    {
        return $sign < 0 && $item->getProduct()->getIsAvailableForBackorder();
    }

    /**
     * @param mixed $shippingStatus Shipping status
     *
     * @return void
     */
    public function setShippingStatus($shippingStatus = null)
    {
        $statusCode = '';
        if (is_numeric($shippingStatus)) {
            /** @var \XLite\Model\Repo\Order\Status\Shipping $repo */
            $repo = \XLite\Core\Database::getRepo('XLite\Model\Order\Status\Shipping');
            $status = $repo->find($shippingStatus);
            $statusCode = $status ? $status->getCode() : '';
        } elseif (is_string($shippingStatus)) {
            $statusCode = $shippingStatus;
        }

        if (
            $statusCode === \XLite\Model\Order\Status\Shipping::STATUS_RETURNED
            && $this->isBackorder()
        ) {
            $this->increaseInventory();
        } elseif (
            in_array(
                $statusCode,
                [\XLite\Model\Order\Status\Shipping::STATUS_BACK_ORDERED, \XLite\Model\Order\Status\Shipping::STATUS_PRE_ORDERED]
            )
            && $this->getShippingStatus()
            && $this->getShippingStatus()->getCode() === \XLite\Model\Order\Status\Shipping::STATUS_RETURNED
        ) {
            $this->decreaseInventory();
        }

        parent::setShippingStatus($shippingStatus);
    }
}
