<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Core;

use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 */
class OrderHistory extends \XLite\Core\OrderHistory
{
    public const CODE_CHANGE_BACKORDERED = 'CHANGE BACKORDERED AMOUNT';

    public const TXT_CHANGE_BACKORDERED = '[Inventory] Backorder/Preorder: "{{product}}" {{itemsBackordered}} of total {{positionQty}} items backordered (were {{oldBackordered}} on backorder).';

    /**
     * @inheritDoc
     */
    public function registerChangeAmount($orderId, $product, $delta)
    {
        // disable built-in registration
        return;
    }

    /**
     * @inheritDoc
     */
    protected function isAbleToReduceAmount($product, $delta)
    {
        if ($product->getIsAvailableForBackorder()) {
            return true; // @todo:  check against backorderLimit + qty in stock
        } else {
            return (int)$product->getPublicAmount() >= 0;
        }
    }

    /**
     * Register the change amount inventory
     *
     * @param  integer                 $orderId         Order identificator
     * @param  \XLite\Model\Product    $product         Product object
     * @param  \XLite\Model\OrderItem  $orderItem       Order item (position)
     * @param  integer                 $oldQty          Product in stock amount before update
     * @param  integer                 $newQty          Product in stock amount after update
     * @param  integer                 $oldBackordered  Order item backordered amount before update
     * @param  integer                 $newBackordered  Order item backordered amount after update
     *
     * @return void
     */
    public function registerChangeAmountForBackordered($orderId, $product, $orderItem, $oldQty, $newQty, $oldBackordered, $newBackordered)
    {
        if ($product->getInventoryEnabled()) {
            if ($oldQty !== $newQty) {
                $delta = $newQty - $oldQty;

                $this->registerEvent(
                    $orderId,
                    static::CODE_CHANGE_AMOUNT,
                    $this->getOrderChangeAmountDescription($orderId, $delta, $product),
                    [
                        'orderId'    => $orderId,
                        'newInStock' => $newQty,
                        'oldInStock' => $oldQty,
                        'product'    => $orderItem->getExtendedItemName(),
                        'qty'        => abs($delta),
                    ]
                );
            }

            if ($oldBackordered !== $newBackordered) {
                $this->registerEvent(
                    $orderId,
                    static::CODE_CHANGE_BACKORDERED,
                    '[Inventory] Backorder/Preorder: "{{product}}" backorder amount changed from "{{oldBackordered}}" to "{{itemsBackordered}}"',
                    [
                        'orderId' => $orderId,
                        'product' => $orderItem->getExtendedItemName(),
                        'positionQty' => $orderItem->getAmount(),
                        'oldBackordered' => $oldBackordered,
                        'itemsBackordered' => $newBackordered,
                    ]
                );
            }
        }
    }

    /**
     * Preprocess data for change amount inventory grouped
     *
     * @param integer              $orderId Order identificator
     * @param array                $data    Inventory changes data
     *
     * @return array
     */
    protected function preprocessForGroupData($orderId, $data)
    {
        if (\XLite::isAdminZone()) {
            return []; // there are conflicts between line-by-line and groupped history records
        }

        $result = [];

        foreach ($data as $item) {
            if (
                $item['item']->getProduct()
                && $item['item']->getProduct()->getInventoryEnabled()
            ) {
                // a bit of adjustment for when partially taken from stock
                // otherwise it would give negative "new stock" in the message:
                //   Ex.: Removed from stock: "Test backorder" product amount in stock changes from "5" to "-5" (10 items)
                $newAmount = $item['amount'] + $item['delta'];
                if ($newAmount < 0) {
                    $item['delta'] = -1 * $item['amount'];
                }

                $newResult = $this->preprocessForGroupDataItem($orderId, $item);
                // replace "Product name" with "Product Name + Options text"
                $productName = $item['item']->getProduct()->getName();
                $productNameExtended = $item['item']->getExtendedItemName();
                $newResult['value'] = str_replace(
                    '"' . $productName . '"',
                    '"' . $productNameExtended . '"',
                    $newResult['value']
                );
                $result[] = $newResult;


                $oldBAmount = $item['item']->getPreviousBackorderAmount();
                $newBAmount = $item['item']->getBackorderAmount();
                if ($oldBAmount !== $newBAmount) {
                    $productName = $item['item']->getExtendedItemName();
                    $result[] = [
                        'name'  => "Backorder/Preorder",
                        'value' => "\"{$productName}\" backorder amount changed from \"{$oldBAmount}\" to \"{$newBAmount}\"",
                        'label' => ''
                    ];
                }
            }
        }

        return $result;
    }
}
