<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Controller\Customer;

use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 */
class Cart extends \XLite\Controller\Customer\Cart
{
    protected function checkAmount(\XLite\Model\OrderItem $item, $amount = null)
    {
        $return = parent::checkAmount($item, $amount)
            || (
                $item->getProduct()->getIsAvailableForBackorder()
                && !$item->getProduct()->getIsBackorderLimit()
            );

        return $return;
    }

    /**
     * Show message about wrong product amount
     *
     * @param \XLite\Model\OrderItem $item Order item
     *
     * @return void
     */
    protected function processInvalidAmountError(\XLite\Model\OrderItem $item)
    {
        if (! $item->getProduct()->isLimitedBackorder()) {
            parent::processInvalidAmountError($item);
            return;
        }

        if ($this->doesItemHaveVariant($item)) {
            $inStock = $this->getItemVariantAmount($item);
            $message  = "You tried to buy more items of \"{{product}}\" product {{description}} than are in stock."
                . " We have {{in_stock}} items of this variation in stock and {{to_backorder}} (in total) available for backorder."
                . " Please adjust the product quantity.";
        } else {
            $inStock = $item->getProduct()->getAmount();
            $message  = "You tried to buy more items of \"{{product}}\" product {{description}} than are in stock."
                . " We have {{in_stock}} items in stock and {{to_backorder}} more available for backorder."
                . " Please adjust the product quantity.";
        }

        \XLite\Core\TopMessage::addWarning(
            $message,
            [
                'product'     => $item->getProduct()->getName(),
                'description' => $item->getExtendedDescription(),
                'in_stock'    => $inStock,
                'to_backorder' => $item->getProduct()->getBackorderLimit(),
            ]
        );
    }

    protected function doesItemHaveVariant($item)
    {
        return method_exists($item, 'getVariant') && $item->getVariant();
    }

    protected function getItemVariantAmount($item)
    {
        $variant = $item->getVariant();

        return $variant->getDefaultAmount()
            ? $item->getProduct()->getAmount()
            : $variant->getAmount();
    }
}
