<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\USPS\View\Model;

use CDev\USPS\Model\Shipping\Processor\USPS;
use XLite\Model\Config;
use XLite\View\FormField\AFormField;
use XLite\View\FormField\Input\Checkbox\OnOff;
use XLite\View\Model\AShippingSettings;

/**
 * USPS configuration form model
 */
class Settings extends AShippingSettings
{
    public const USPS_PROCESSOR_ID = 'usps';
    public const USPS_DATA_PROVIDER = 'USPS';

    /**
     * @return array
     */
    public function getCSSFiles()
    {
        $list   = parent::getCSSFiles();
        $list[] = 'modules/CDev/USPS/style.css';

        return $list;
    }

    /**
     * Detect form field class by option
     *
     * @param Config $option Option
     *
     * @return string
     */
    protected function detectFormFieldClassByOption(Config $option)
    {
        return $option->getName() === 'dimensions'
            ? 'XLite\View\FormField\Input\Text\Dimensions'
            : parent:: detectFormFieldClassByOption($option);
    }

    /**
     * Get form field by option
     *
     * @param Config $option Option
     *
     * @return array
     */
    protected function getFormFieldByOption(Config $option)
    {
        $cell = parent::getFormFieldByOption($option);

        switch ($option->getName()) {
            case 'cod_status':
                $cell[OnOff::PARAM_DISABLED]  = true;
                $cell[OnOff::PARAM_ON_LABEL]  = static::t('paymentStatus.Active');
                $cell[OnOff::PARAM_OFF_LABEL] = static::t('paymentStatus.Inactive');
                $cell[static::SCHEMA_COMMENT] = static::t(
                    'usps.CODStatusOptionComment',
                    [
                        'URL' => $this->buildURL('payment_settings'),
                    ]
                );
                break;

            case 'cod_price':
                $cell[static::SCHEMA_DEPENDENCY] = [
                    static::DEPENDENCY_SHOW => [
                        'use_cod_price' => [true],
                    ],
                ];
                break;

            case 'first_class_mail_type':
                $cell[static::SCHEMA_DEPENDENCY] = [
                    static::DEPENDENCY_SHOW => [
                        'use_cod_price' => [false],
                    ],
                ];
                break;

            case 'gxg_pobox':
            case 'gxg_gift':
                $cell[static::SCHEMA_DEPENDENCY] = [
                    static::DEPENDENCY_SHOW => [
                        'gxg' => [true],
                    ],
                ];
                break;

            case 'dimensions':
                $value = $this->getModelObjectValue('dimensions');
                foreach ($value as $dimension) {
                    if ((int) $dimension === 0) {
                        $cell[static::SCHEMA_COMMENT] = static::t('All dimensions must be greater than 0');
                        break;
                    }
                }
                break;
        }

        return $cell;
    }

    /**
     * Retrieve property from the model object
     *
     * @param mixed $name Field/property name
     *
     * @return mixed
     */
    protected function getModelObjectValue($name)
    {
        $value = parent::getModelObjectValue($name);
        if ($name === 'dimensions') {
            $value = unserialize($value, ['allowed_classes' => false]);
        }

        return $value;
    }

    /**
     * Get editable options
     *
     * @return array
     */
    protected function getEditableOptions()
    {
        $list = parent::getEditableOptions();

        foreach ($list as $k => $option) {
            if ($option->getName() === 'cod_status') {
                unset($list[$k]);
            } elseif ($option->getName() === 'first_class_mail_type' && !USPS::isCODPaymentEnabled()) {
                unset($list[$k]);
            }
        }

        return $list;
    }

    /**
     * Validate 'container' field (domestic).
     * Return array (<bool: isValid>, <string: error message>)
     *
     * @param AFormField $field Form field object
     * @param array      $data  List of all fields
     *
     * @return array
     */
    protected function validateFormFieldContainerValue($field, $data)
    {
        $errorMessage = null;

        // Get domestic container type value
        $container = $field->getValue();

        // Get package size value
        $packageSize = isset($data['commonOptionsSeparator'][static::SECTION_PARAM_FIELDS]['package_size'])
            ? $data['commonOptionsSeparator'][static::SECTION_PARAM_FIELDS]['package_size']->getValue()
            : null;

        $isMultiRequests = false;

        if (USPS::isCODPaymentEnabled()) {
            $isMultiRequests = isset($data['cacheOnDeliverySeparator'][static::SECTION_PARAM_FIELDS]['use_cod_price'])
                ? !(bool) $data['cacheOnDeliverySeparator'][static::SECTION_PARAM_FIELDS]['use_cod_price']->getValue()
                : false;
        }

        if ($packageSize === 'LARGE') {
            if (!in_array($container, ['VARIABLE', 'RECTANGULAR', 'NONRECTANGULAR'])) {
                $errorMessage = static::t(
                    'Wrong container type selected: {{value}}. For large package size only the following types are allowed: RECTANGULAR, NONRECTANGULAR, VARIABLE',
                    ['value' => $container]
                );
            }
        }

        if (!$isMultiRequests) {
            // Service ONLINE has limited types of Container values
            if ($packageSize === 'REGULAR' && $container !== 'VARIABLE') {
                $errorMessage = static::t(
                    '{{value}} is an invalid container type for a REGULAR package. Valid Container is: VARIABLE',
                    ['value' => $container]
                );
            }
        }

        return [empty($errorMessage), $errorMessage];
    }

    /**
     * Populate model object properties by the passed data
     *
     * @param array $data Data to set
     *
     * @return void
     */
    protected function setModelProperties(array $data)
    {
        if ($this->isValid()) {
            parent::setModelProperties($data);
        }
    }

    /**
     * Return true if specific section is collapsible
     *
     * @param string $section
     *
     * @return boolean
     */
    protected function isSectionCollapsible($section)
    {
        return false;
    }
}
