<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\USPS\Controller\Admin;

use CDev\USPS\View\Model\Settings;
use XLite\Controller\Admin\ShippingSettings;
use XLite\InjectLoggerTrait;
use XLite\Model\Config;
use XLite\Model\Shipping\Processor\AProcessor;

/**
 * USPS module settings page controller
 */
class Usps extends ShippingSettings
{
    use InjectLoggerTrait;

    /**
     * Returns shipping options
     *
     * @return Config[]
     */
    public function getOptions()
    {
        $list = [];

        $CODRelatedOptions = ['first_class_mail_type', 'use_cod_price', 'cod_price'];
        foreach (parent::getOptions() as $option) {
            if (
                !in_array($option->getName(), $CODRelatedOptions, true)
                || $this->isUSPSCODPaymentEnabled()
            ) {
                $list[] = $option;
            }

            if ($option->getName() === 'cacheOnDeliverySeparator') {
                $list[] = new Config([
                    'name'        => 'cod_status',
                    'type'        => 'XLite\View\FormField\Input\Checkbox\OnOff',
                    'value'       => $this->isUSPSCODPaymentEnabled() ? true : false,
                    'orderby'     => $option->getOrderby() + 1,
                    'option_name' => static::t('"Cash on delivery" status'),
                ]);
            }
        }

        return $list;
    }

    /**
     * getOptionsCategory
     *
     * @return string
     */
    protected function getOptionsCategory()
    {
        return 'CDev\USPS';
    }

    /**
     * Class name for the \XLite\View\Model\ form (optional)
     *
     * @return string
     */
    protected function getModelFormClass()
    {
        return 'CDev\USPS\View\Model\Settings';
    }

    /**
     * Get shipping processor
     *
     * @return AProcessor
     */
    protected function getProcessor()
    {
        return new \CDev\USPS\Model\Shipping\Processor\USPS();
    }

    /**
     * Returns current processor id
     *
     * @return string
     */
    public function getProcessorId()
    {
        return Settings::USPS_PROCESSOR_ID;
    }

    /**
     * Check if 'Cash on delivery (FedEx)' payment method enabled
     *
     * @return boolean
     */
    protected function isUSPSCODPaymentEnabled()
    {
        return \CDev\USPS\Model\Shipping\Processor\USPS::isCODPaymentEnabled();
    }

    /**
     * Get schema of an array for test rates routine
     *
     * @return array
     */
    protected function getTestRatesSchema()
    {
        $schema = parent::getTestRatesSchema();

        foreach (['srcAddress', 'dstAddress'] as $k) {
            unset($schema[$k]['city'], $schema[$k]['state']);
        }

        unset($schema['dstAddress']['type']);

        return $schema;
    }

    /**
     * Get input data to calculate test rates
     *
     * @param array $schema  Input data schema
     * @param array &$errors Array of fields which are not set
     *
     * @return array
     */
    protected function getTestRatesData(array $schema, &$errors)
    {
        $data = parent::getTestRatesData($schema, $errors);

        $config = \XLite\Core\Config::getInstance()->CDev->USPS;

        $package = [
            'weight'   => $data['weight'],
            'subtotal' => $data['subtotal'],
            'length'   => $config->dimensions[0],
            'width'    => $config->dimensions[1],
            'height'   => $config->dimensions[2],
        ];

        $data['packages'] = [$package];

        unset($data['weight'], $data['subtotal']);

        return $data;
    }

    /**
     * Refresh list of available USPS shipping methods
     *
     * @return void
     */
    protected function doActionRefresh()
    {
        $config = \XLite\Core\Config::getInstance()->CDev->USPS;

        // Prepare default input data
        $data               = [];
        $data['packages']   = [];
        $data['packages'][] = [
            'weight'   => 5,
            'subtotal' => 50,
            'length'   => $config->length,
            'width'    => $config->width,
            'height'   => $config->height,
        ];
        $data['srcAddress'] = [
            'country' => 'US',
            'zipcode' => '10001',
        ];

        // Prepare several destination addresses
        $dstAddresses   = [];
        $dstAddresses[] = [
            'country' => 'US',
            'zipcode' => '90001',
        ];
        $dstAddresses[] = [
            'country' => 'CA',
            'zipcode' => 'V7P 1S0',
        ];
        $dstAddresses[] = [
            'country' => 'GB',
            'zipcode' => 'EC1A 1BB',
        ];
        $dstAddresses[] = [
            'country' => 'CN',
            'zipcode' => '100001',
        ];

        foreach ($dstAddresses as $addr) {
            $data['dstAddress'] = $addr;

            // Get rates for each destination address.
            // All non-existing methods will be created after this
            $this->getProcessor()->getRates($data, true);
        }

        $this->setReturnURL(
            $this->buildURL('shipping_methods', null, ['processor' => Settings::USPS_PROCESSOR_ID])
        );
    }
}
