/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Logo SVG to PNG on-the-fly conversion.
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

document.addEventListener('DOMContentLoaded', () => {

  /**
   * Detect the logo width and height.
   *
   * @return {{logoHeight: number, logoWidth: number}}
   */
  const getLogoImageSizes = () => {
    let result = {
      logoWidth: 0,
      logoHeight: 0
    };

    const wrapper = document.querySelector('.notificationlogo-value .file-uploader-wrapper');

    if (wrapper) {
      const newWidth = wrapper.getAttribute('data-logo-width');
      const newHeight = wrapper.getAttribute('data-logo-height');

      result = {
        logoWidth: newWidth ? newWidth : 0,
        logoHeight: newHeight ? newHeight : 0
      };
    }

    return result;
  };

  /**
   * Save autogenerated PNG image into the <input type="file" />
   *
   * @param {HTMLCanvasElement} canvas
   * @param {number} width
   * @param {number} height
   * @param {string} filename
   */
  const saveAutogeneratedNotificationLogo = (canvas, width, height, filename) => {
    canvas.toBlob(
      (blob) => {
        const dataTransfer = new DataTransfer();
        dataTransfer.items.add(
          new File(
            [blob],
            filename.replace(/\.[^/.]+$/, '') + '.png', // replace the real extension with the '.png'
            {type: 'image/png'}
          )
        );

        const form = document.querySelector('.general-settings > form');
        let file = form.querySelector('.generated-png');

        if (!file) {
          file = document.createElement('INPUT');
          file.type = 'file';
          file.className = 'generated-png';
          file.name = 'notification-logo-generated';
          file.setAttribute('style', 'display:none');
          form.insertBefore(file, form.firstElementChild);
        }

        file.files = dataTransfer.files;
      },
      'image/png',
      .95
    );
  };

  xcart.bind(
    'image-upload-contents',
    function (event, {file, content, base}) {

      if (base.parentNode.parentNode.classList.contains('logo-value')) {
        const img = new Image();
        const {width, height} = xcart.utils.getSvgImageSizes(content);

        if (width > 0 && height > 0) {
          const {logoWidth, logoHeight} = getLogoImageSizes();
          let rate = 0.0;

          if (logoWidth || logoHeight) {
            if (logoWidth) {
              rate = logoWidth / width;
            }

            if (logoHeight) {
              rate = Math.max(rate, logoHeight / height);
            }
          }

          if (rate <= 0) {
            rate = 2;
          }

          img.onload = () => {
            const canvas = document.createElement('CANVAS');
            canvas.width = width * rate;
            canvas.height = height * rate;
            const context = canvas.getContext('2d');
            context.imageSmoothingEnabled = context.webkitImageSmoothingEnabled = false;
            context.drawImage(img, 0, 0, width * rate, height * rate);
            saveAutogeneratedNotificationLogo(canvas, width, height, file.name);
          };

          img.src = URL.createObjectURL(
            new Blob([content], {type: 'image/svg+xml;charset=utf-8'})
          );
        }
      }
    });
});
