<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\Sale\Logic;

use CDev\Sale\Model\SaleDiscount as SaleDiscountEntity;
use CDev\Sale\View\FormField\Select\CombineDiscounts;
use XLite\Core\Database;

/**
 * Net price modificator: price with sale discount
 */
class SaleDiscount extends \XLite\Logic\ALogic
{
    /**
     * Check modificator - apply or not
     *
     * @param \XLite\Model\AEntity $model     Model
     * @param string               $property  Model's property
     * @param array                $behaviors Behaviors
     * @param string               $purpose   Purpose
     *
     * @return boolean
     */
    public static function isApply(\XLite\Model\AEntity $model, $property, array $behaviors, $purpose)
    {
        return self::getObject($model) instanceof \XLite\Model\Product
            && !self::getObject($model)->getParticipateSale();
    }

    /**
     * Modify money
     *
     * @param \XLite\Model\AEntity $model Model
     *
     * @return \XLite\Model\AEntity
     */
    protected static function getObject(\XLite\Model\AEntity $model)
    {
        return $model instanceof \XLite\Model\Product ? $model : $model->getProduct();
    }

    /**
     * Modify money
     *
     * @param float                $value     Value
     * @param \XLite\Model\AEntity $model     Model
     * @param string               $property  Model's property
     * @param array                $behaviors Behaviors
     * @param string               $purpose   Purpose
     *
     * @return float
     */
    public static function modifyMoney($value, \XLite\Model\AEntity $model, $property, array $behaviors, $purpose)
    {
        $saleValue = static::getSaleDiscountValue($model, $value);

        $result = $value - $saleValue;

        $currency = \XLite::getInstance()->getCurrency();

        if ($currency) {
            $result = \XLite::getInstance()->getCurrency()->roundValue($result);
        }

        return $result;
    }

    protected static function getSaleDiscountValue($model, $value)
    {
        return $value * min(100, static::getSaleDiscountPercentage($model)) / 100;
    }

    public static function getSaleDiscountPercentage(\XLite\Model\AEntity $model): int
    {

        return match (\XLite\Core\Config::getInstance()->CDev->Sale->way_to_combine_discounts) {
            CombineDiscounts::TYPE_APPLY_MAX => static::getMaxSaleDiscountPercentage($model),
            CombineDiscounts::TYPE_APPLY_MIN => static::getMinSaleDiscountPercentage($model),
            CombineDiscounts::TYPE_SUM_UP => static::getSumSaleDiscountPercentage($model),
            default => 0,
        };
    }

    protected static function getMinSaleDiscountPercentage(\XLite\Model\AEntity $model): int
    {
        $activeSaleDiscounts = static::getActiveSaleDiscounts();

        for ($i = count($activeSaleDiscounts) - 1; $i > 0; $i--) {
            $saleDiscount = $activeSaleDiscounts[$i];

            if (static::isSaleDiscountApplicable($saleDiscount, $model)) {
                return $saleDiscount->getValue();
            }
        }

        return 0;
    }

    protected static function getMaxSaleDiscountPercentage(\XLite\Model\AEntity $model): int
    {
        foreach (static::getActiveSaleDiscounts() as $saleDiscount) {
            if (static::isSaleDiscountApplicable($saleDiscount, $model)) {
                return $saleDiscount->getValue();
            }
        }

        return 0;
    }

    protected static function getSumSaleDiscountPercentage(\XLite\Model\AEntity $model): int
    {
        $result = 0;

        foreach (static::getActiveSaleDiscounts() as $saleDiscount) {
            if (static::isSaleDiscountApplicable($saleDiscount, $model)) {
                $result += $saleDiscount->getValue();
            }
        }

        return $result;
    }

    protected static function getActiveSaleDiscounts(): array
    {
        return Database::getRepo(SaleDiscountEntity::class)->findAllActiveForCalculate();
    }

    protected static function isSaleDiscountApplicable(SaleDiscountEntity $saleDiscount, $model)
    {
        $object = static::getObject($model);
        if (!$object instanceof \XLite\Model\Product) {
            $object = $object->getProduct();
        }

        if (!$saleDiscount->isApplicableForProduct($object)) {
            return false;
        }

        $controller = \XLite::getController();
        $profile = null;

        if ($controller instanceof \XLite\Controller\Customer\ACustomer) {
            $profile = $controller->getCart(true)->getProfile()
                ?: \XLite\Core\Auth::getInstance()->getProfile();
        }

        if (!$profile) {
            $profile = new \XLite\Model\Profile();
        }

        if (!$saleDiscount->isApplicableForProfile($profile)) {
            return false;
        }

        return true;
    }
}
