<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\Paypal\View;

use CDev\Paypal;
use CDev\Paypal\View\FormField\Select\PPCMBannerType;
use XCart\Domain\ModuleManagerDomain;

/**
 * Paypal credit messaging banner
 */
class PPCMBanner extends \XLite\View\AView
{
    public const PARAM_PAGE = 'page';

    public const PAGE_PRODUCT      = 'product';
    public const PAGE_CART         = 'cart';
    public const PAGE_CHECKOUT     = 'checkout';
    public const PAGE_OPC_CHECKOUT = 'opc_checkout';
    public const PAGE_FLC_CHECKOUT = 'flc_checkout';

    protected ?ModuleManagerDomain $moduleManagerDomain;

    public function __construct(array $params = [])
    {
        $this->moduleManagerDomain = \XCart\Container::getContainer()?->get(ModuleManagerDomain::class);

        parent::__construct($params);
    }

    /**
     * Get css files
     *
     * @return array
     */
    public function getCSSFiles()
    {
        $list   = parent::getCSSFiles();
        $list[] = 'modules/CDev/Paypal/ppcm_banner/style.less';

        return $list;
    }

    /**
     * @return array
     */
    public function getJSFiles()
    {
        $list   = parent::getJSFiles();
        $list[] = 'modules/CDev/Paypal/ppcm_banner/controller.js';

        return $list;
    }

    /**
     * Define widget parameters
     *
     * @return void
     */
    protected function defineWidgetParams()
    {
        parent::defineWidgetParams();

        $this->widgetParams += [
            static::PARAM_PAGE => new \XLite\Model\WidgetParam\TypeString('Page', ''),
        ];
    }

    /**
     * Get payment method
     *
     * @return \XLite\Model\Payment\Method
     */
    protected function getMethod()
    {
        return Paypal\Main::getPaymentMethod(Paypal\Main::PP_METHOD_PC);
    }

    /**
     * Get Paypal credit setting
     *
     * @param string $name Setting name
     *
     * @return string
     */
    protected function getSetting($name)
    {
        return $this->getMethod()->getSetting($name);
    }

    /**
     * @return string
     */
    protected function getBannerAttributes()
    {
        $attributes = [
            'data-pp-placement'    => $this->getPlacementValue(),
            'data-pp-style-layout' => $this->getSetting('ppcm_banner_type'),
        ];

        $textStyleAttributes = [
            'data-pp-style-logo-type'     => $this->getSetting('ppcm_text_logo_type'),
            'data-pp-style-logo-position' => $this->getSetting('ppcm_text_logo_position'),
            'data-pp-style-text-size'     => $this->getSetting('ppcm_text_size'),
            'data-pp-style-text-color'    => $this->getSetting('ppcm_text_color'),
        ];

        $flexStyleAttributes = [
            'data-pp-style-color' => $this->getSetting('ppcm_flex_color_scheme'),
            'data-pp-style-ratio' => $this->getSetting('ppcm_flex_layout'),
        ];

        if ($this->getSetting('ppcm_banner_type') === PPCMBannerType::PPCM_FLEX) {
            $attributes = array_merge($attributes, $flexStyleAttributes);
        } else {
            $attributes = array_merge($attributes, $textStyleAttributes);
        }

        $result = '';
        foreach ($attributes as $attr => $value) {
            $result .= ' ' . $attr;

            if ($value !== '') {
                $result .= '="' . $value . '"';
            }
        }

        return $result;
    }

    /**
     * @return string
     */
    protected function getPlacementValue()
    {
        switch ($this->getParam(static::PARAM_PAGE)) {
            case static::PAGE_PRODUCT:
                return 'product';

            case static::PAGE_CART:
                return 'cart';

            case static::PAGE_CHECKOUT:
            case static::PAGE_OPC_CHECKOUT:
            case static::PAGE_FLC_CHECKOUT:
                return 'payment';

            default:
                return '';
        }
    }

    /**
     * Check if widget is visible
     *
     * @return boolean
     */
    protected function isVisible()
    {
        $result = parent::isVisible()
            && Paypal\Main::isPaypalCreditForCommercePlatformEnabled()
            && $this->getSetting('ppcm_enabled');

        $isFlcEnabled = $this->moduleManagerDomain->isEnabled('XC-FastLaneCheckout')
            && \XC\FastLaneCheckout\Main::isFastlaneEnabled();

        switch ($this->getParam(static::PARAM_PAGE)) {
            case static::PAGE_PRODUCT:
                return $result && $this->getSetting('ppcm_product_page');

            case static::PAGE_CART:
                return $result && $this->getSetting('ppcm_cart_page');

            case static::PAGE_CHECKOUT:
                return $result && $this->getSetting('ppcm_checkout_page');

            case static::PAGE_OPC_CHECKOUT:
                return $result
                    && $this->getSetting('ppcm_checkout_page')
                    && !$isFlcEnabled;

            case static::PAGE_FLC_CHECKOUT:
                return $result
                    && $this->getSetting('ppcm_checkout_page')
                    && $isFlcEnabled;

            default:
                return false;
        }
    }

    /**
     * Return default template
     *
     * @return string
     */
    protected function getDefaultTemplate()
    {
        return 'modules/CDev/Paypal/ppcm_banner/body.twig';
    }
}
