<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\Paypal\Core;

/**
 * RESTAPI
 *
 * @see https://developer.paypal.com/docs/api/
 */
class RESTAPI extends \CDev\Paypal\Core\AAPI
{
    /**
     * API credentials
     * (vab@x-cart.com)
     */
    public const CLIENT_ID     = 'AWTzQBBzsLZufFGNl_oWWdzM7BqB27aLXw2SRUYGb4U-Qi104Db5tF0OPnRg';
    public const CLIENT_SECRET = 'EEQdxRAZua_WrdOeY9Yl5vzraRwDerCMSkHoc_q89PAArqK6Gs8kcXQT5Weq';
    public const PARTNER_ID    = 'FWY6M72RRKFGW';

    // {{{ Config

    /**
     * Is test mode
     *
     * @return boolean
     */
    public function isTestMode()
    {
        return false;
    }

    // }}}

    // {{{ Sign up

    /**
     * Is In-Context Boarding SignUp available
     *
     * @return boolean
     */
    public function isInContextSignUpAvailable()
    {
        return in_array(
            \XLite\Core\Config::getInstance()->Company->location_country,
            ['US', 'GB']
        );
    }

    /**
     * Get SignUp url
     *
     * @param string $returnUrl Return url
     *
     * @return string
     */
    public function getSignUpUrl($returnUrl)
    {
        $urlParams = [
            // Your secure merchant account ID, also known as your payer ID. This
            // can be obtained from your PayPal account.
            'partnerId' => static::PARTNER_ID,

            // The product you want the merchant to enroll for. At this time, only
            // the values shown below are supported; however, as this product
            // matures, additional products will be added to this list.
            // - addipmt (Express Checkout)
            // - wp_pro (PayPal Payments Pro)
            'productIntentID' => 'addipmt',

            // The merchant’s country of residence.
            // - US
            // - UK
            'countryCode' => \XLite\Core\Config::getInstance()->Company->location_country,

            // Indicates how you intend to display Integrated PayPal Signup to the
            // merchant. This can be either regular or minibrowser. (See
            // section X.X for information on displaying Integrated PayPal Signup
            // to the merchant in a minibrowser.)
            'displayMode' => 'minibrowser',

            // Indicates whether you are requesting first-party API credentials (F)
            // or third party API permissions (T).
            'integrationType' => 'F',

            // If you are requesting first-party API credentials, this indicates
            // whether you are requesting credentials that contain an API
            // signature (S) or an API certificate (C).
            // Note: If you are retrieving the merchant’s API credentials through
            // the REST API (see section X.X), the API currently only supports
            // retrieval of credentials that contain an API signature.
            'subIntegrationType' => 'S',

            // A comma-separated list of API permissions that need to be granted
            // from the merchant’s account to yours. This field is only required if
            // you are requesting third party permissions from the merchant.
            'permissionNeeded' => 'EXPRESS_CHECKOUT,REFUND,AUTH_CAPTURE',

            // URL where PayPal will return the merchant after they have
            // completed the signup flow on PayPal.
            'returnToPartnerUrl' => $returnUrl,

            // Indicates whether you want to receive the merchant’s API
            // credentials (TRUE) or whether you want PayPal to simply display
            // them to the merchant (FALSE). If you are requesting third-party
            // permissions to the merchant’s account, set this to FALSE. If you
            // do not specify a value for this field, PayPal does not provision the
            // merchant with API credentials.
            'receiveCredentials' => 'TRUE',

            // Indicates whether you need the merchant to grant third-party API
            // permissions to you. If you are signing up a merchant from your
            // front-of-site, you should generally set this to FALSE.
            'showPermissions' => 'FALSE',

            'productSelectionNeeded' => 'FALSE',

            // A unique identifier that you generate for the merchant. This will be
            // passed back to you when the merchant returns to you from PayPal.
            // If you are requesting first-party API credentials for the merchant
            // (e.g., you set receiveCredentials to TRUE), you will use this
            // identifier to request API credentials from PayPal later.
            'merchantId' => 'TEST0001',

            'partnerLogoUrl' => urlencode(\CDev\Paypal\Main::getSignUpLogo()),
        ];

        return 'https://www.paypal.com/webapps/merchantboarding/webflow/externalpartnerflow'
        . '?&'
        . http_build_query($urlParams, null, '&');
    }

    // }}}

    // {{{ Access token

    /**
     * Retrieve access token
     * todo: caching
     *
     * @param string $clientId     Client id
     * @param string $clientSecret Client secret
     *
     * @return mixed
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    public function getAccessToken($clientId, $clientSecret)
    {
        $params = [$clientId, $clientSecret];

        return $this->doRequest('accessToken', $params);
    }

    /**
     * Prepare url
     *
     * @param string $url    Request url
     * @param array  $params Request params
     *
     * @return string
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareAccessTokenUrl($url, $params)
    {
        return $url . '/v1/oauth2/token';
    }

    /**
     * Prepare body
     *
     * @param array $params Request params
     *
     * @return string
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareAccessTokenParams($params)
    {
        return ['grant_type' => 'client_credentials'];
    }

    /**
     * @param array $requestData
     * @param array $params Request params
     *
     * @return array
     * @see https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareAccessTokenRequestData($requestData, $params)
    {
        $requestData['options']['headers']['Accept'] = 'application/json';
        $requestData['options']['headers']['Accept-Language'] = 'en_US';
        $requestData['options']['headers']['Content-Type'] = 'application/x-www-form-urlencoded';
        $requestData['options']['headers']['Authorization'] = sprintf('Basic %s', base64_encode(implode(':', $params)));

        return $requestData;
    }

    // }}}

    // {{{ Merchant credentials

    /**
     * Retrieve merchant credentials
     *
     * @param string $partnerId  Partner id
     * @param string $merchantId Merchant id
     *
     * @return mixed
     */
    public function getMerchantCredentials($partnerId, $merchantId)
    {
        $data = [$partnerId, $merchantId];

        return $this->doRequest('merchantCredentials', $data);
    }

    /**
     * Prepare url
     *
     * @param string $url    Request url
     * @param array  $params Request params
     *
     * @return string
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareMerchantCredentialsUrl($url, $params)
    {
        [$partnerId, $merchantId] = $params;

        return sprintf('%s/v1/customer/partners/%s/merchant-integrations/%s', $url, $partnerId, $merchantId);
    }

    /**
     * @param array $requestData
     * @param array $params Request params
     *
     * @return array
     */
    protected function prepareMerchantCredentialsRequestData($requestData, $params)
    {
        $requestData['method'] = 'GET';

        $accessToken = $this->getAccessToken(static::CLIENT_ID, static::CLIENT_SECRET);
        if (is_array($accessToken) && $accessToken['access_token']) {
            $requestData['options']['headers']['Authorization'] = sprintf('Bearer %s', $accessToken['access_token']);
        }

        return $requestData;
    }

    /**
     * Prepare body
     *
     * @param array $params Request params
     *
     * @return string
     */
    protected function prepareMerchantCredentialsParams($params)
    {
        return [];
    }

    // }}}

    // {{{ Backend request

    /**
     * Prepare url
     *
     * @param string $url    Url
     * @param string $type   Request type
     * @param array  $params Request params
     *
     * @return string
     */
    protected function prepareUrl($url, $type, $params)
    {
        $url = $this->isTestMode()
            ? 'https://api.sandbox.paypal.com'
            : 'https://api.paypal.com';

        return parent::prepareUrl($url, $type, $params);
    }

    /**
     * Returns parsed response
     *
     * @param string $type Response type
     * @param string $body Response body
     *
     * @return array
     */
    protected function parseResponse($type, $body)
    {
        return json_decode($body, true);
    }
}
