<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\Paypal\Core\PaypalCommercePlatform;

use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use XLite\InjectLoggerTrait;

class AccessToken
{
    use InjectLoggerTrait;

    public const SECURITY_HEADER = 'XCARTCDEVPAYPAL';

    /**
     * @var array
     */
    protected $accessTokenData;

    protected bool $sandbox;

    protected HttpClientInterface $httpClient;

    public function __construct(bool $sandbox, HttpClientInterface $httpClient)
    {
        $this->sandbox = $sandbox;
        $this->httpClient = $httpClient;
    }

    /**
     * @return array {
     *      scope: string,
     *      access_token: string,
     *      token_type: string,
     *      app_id: string,
     *      expires_in: int,
     *      nonce: string,
     *      expiration: int,
     *      partner_id: string,
     * }
     */
    public function getAccessTokenData()
    {
        if ($this->accessTokenData === null) {
            $accessTokenData = \XLite\Core\Cache\ExecuteCached::getCache(['\CDev\Paypal\Core\PaypalCommercePlatform\AccessToken::getAccessTokenData']);
            if (empty($accessTokenData)) {
                $accessTokenData = $this->retrieveAccessTokenData();

                if ($accessTokenData) {
                    \XLite\Core\Cache\ExecuteCached::setCache(
                        ['\CDev\Paypal\Core\PaypalCommercePlatform\AccessToken::getAccessTokenData'],
                        $accessTokenData,
                        3600
                    );
                }
            }

            $this->accessTokenData = $accessTokenData;
        }

        return $this->accessTokenData;
    }

    /**
     * @return array
     */
    protected function retrieveAccessTokenData(): array
    {
        try {
            $this->getLogger('CDev-Paypal')->debug('PaypalCommercePlatform Onboarding AccessToken: Retrieve access token');

            $response = $this->httpClient->request(
                'POST',
                $this->getAccessTokenUrl(),
                [
                    'headers' => [
                        self::SECURITY_HEADER => hash('sha256', self::SECURITY_HEADER)
                    ]
                ]
            );

            $responseBody = $response->getContent(false);
            $responseStatusCode = $response->getStatusCode();
            if ($responseStatusCode === 200 && !empty($responseBody)) {
                $this->getLogger('CDev-Paypal')->debug('PaypalCommercePlatform Onboarding AccessToken: Access token recieved');
                return @json_decode($responseBody, true);
            }
        } catch (TransportExceptionInterface) {
        }

        return [];
    }

    /**
     * @return string
     */
    protected function getAccessTokenUrl(): string
    {
        return $this->isSandbox()
            ? 'https://mc-end-auth.qtmsoft.com/paypal-access-token.php?sandbox'
            : 'https://mc-end-auth.qtmsoft.com/paypal-access-token.php';
    }

    /**
     * @return bool
     */
    protected function isSandbox(): bool
    {
        return $this->sandbox;
    }
}
