<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\Paypal\Core;

/**
 * Login
 */
class Login extends \CDev\Paypal\Core\RESTAPI
{
    // {{{ Config

    /**
     * Check configuration
     *
     * @return boolean
     */
    public static function isConfigured()
    {
        return \Xlite\Core\Config::getInstance()->CDev->Paypal->loginClientId
            && \Xlite\Core\Config::getInstance()->CDev->Paypal->loginClientSecret;
    }

    /**
     * Check enabled connect with Pay-Pal
     *
     * @return bool
     */
    public static function isEnabledConnect()
    {
        return \Xlite\Core\Config::getInstance()->CDev->Paypal->connectWithPayPal === 'enabled';
    }

    /**
     * Is test mode
     *
     * @return boolean
     */
    public function isTestMode()
    {
        return \Xlite\Core\Config::getInstance()->CDev->Paypal->loginMode !== 'live';
    }

    /**
     * Check request
     *
     * @return boolean
     */
    public function checkRequest()
    {
        return (bool) \XLite\Core\Request::getInstance()->code
            && \XLite\Core\Request::getInstance()->auth_provider == 'PayPal';
    }

    /**
     * Returns scope
     *
     * @return array
     */
    protected function getScope()
    {
        $result = \XLite\Core\Config::getInstance()->CDev->Paypal->loginScopes;

        if ($result) {
            $result = @unserialize($result, ['allowed_classes' => false]);
        }

        return is_array($result)
            ? $result
            : [ 'openid' ];
    }

    // }}}

    // {{{ SignIn

    /**
     * Get SignIn url
     *
     * @return string
     */
    public function getSignInURL()
    {
        $url = $this->isTestMode()
            ? 'https://www.sandbox.paypal.com/webapps/auth/protocol/openidconnect'
            : 'https://www.paypal.com/webapps/auth/protocol/openidconnect';

        $params = [
            'client_id'     => \Xlite\Core\Config::getInstance()->CDev->Paypal->loginClientId,
            'response_type' => 'code',
            'scope'         => implode(' ', $this->getScope()),
            'redirect_uri'  => $this->getSignInReturnURL(),
            'state' => \XLite\Core\Request::getInstance()->state
        ];

        return sprintf('%s/v1/authorize?%s', $url, http_build_query($params, null, '&', PHP_QUERY_RFC3986));
    }

    /**
     * Get value for redirect_uri parameter
     *
     * @param boolean $isHTTPS Flag: true - use https protocol, false - http
     *
     * @return string
     */
    public function getSignInReturnURL($isHTTPS = null)
    {
        if (is_null($isHTTPS)) {
            $isHTTPS = (\XLite\Core\Config::getInstance()->CDev->Paypal->loginRedirectURLProtocol == 'https');
        }

        $url = \XLite::getInstance()->getShopURL(
            \XLite\Core\Converter::buildURL(
                'paypal_login',
                'login',
                ['auth_provider' => 'PayPal'],
                '',
                false,
                false
            ),
            $isHTTPS
        );

        $sessionId = \XLite\Core\Session::getInstance()->getName();

        return preg_replace('/(&' . preg_quote($sessionId) . '=[^&]*)/', '', $url);
    }

    // }}}

    // {{{ Token (from authorisation code)

    /**
     * Retrieve user info token
     *
     * @param string $code Code
     *
     * @return mixed
     */
    public function createFromAuthorisationCode($code)
    {
        $params = ['code' => $code];

        return $this->doRequest('createFromAuthorisationCode', $params);
    }

    /**
     * Prepare url
     *
     * @param string $url    Request url
     * @param array  $params Request params
     *
     * @return string
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareCreateFromAuthorisationCodeUrl($url, $params)
    {
        return $url . '/v1/identity/openidconnect/tokenservice';
    }

    /**
     * Prepare body
     *
     * @param array $params Request params
     *
     * @return string
     */
    protected function prepareCreateFromAuthorisationCodeParams($params)
    {
        return $params + ['grant_type' => 'authorization_code'];
    }

    // }}}

    // {{{ Token (from refresh token)

    /**
     * Retrieve user info token
     *
     * @param string $token Refresh token
     *
     * @return mixed
     */
    public function createFromRefreshToken($token)
    {
        $params = ['refresh_token' => $token];

        return $this->doRequest('createFromAuthorisationCode', $params);
    }

    /**
     * Prepare url
     *
     * @param string $url    Request url
     * @param array  $params Request params
     *
     * @return string
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareCreateFromRefreshTokenUrl($url, $params)
    {
        return $url . '/v1/identity/openidconnect/tokenservice';
    }

    /**
     * Prepare body
     *
     * @param array $params Request params
     *
     * @return string
     */
    protected function prepareCreateFromRefreshTokenParams($params)
    {
        return $params + [
            'grant_type' => 'refresh_token',
            'scope'      => $this->getScope()
        ];
    }

    // }}}

    // {{{ User info

    /**
     * Retrieve user info
     *
     * @param string $accessToken Access token
     *
     * @return mixed
     */
    public function getUserinfo($accessToken)
    {
        $params = [$accessToken];

        return $this->doRequest('getUserinfo', $params);
    }

    /**
     * Prepare url
     *
     * @param string $url    Request url
     * @param array  $params Request params
     *
     * @return string
     * @see    https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareGetUserinfoUrl($url, $params)
    {
        return $url . '/v1/identity/openidconnect/userinfo?schema=openid';
    }

    /**
     * @param array $requestData
     * @param array $params Request params
     *
     * @return array
     * @see https://developer.paypal.com/docs/api/#authentication--headers
     */
    protected function prepareGetUserinfoRequestData($requestData, $params)
    {
        [$accessToken] = $params;

        $requestData['options']['headers']['Authorization'] = sprintf('Bearer %s', $accessToken);
        $requestData['method'] = 'GET';

        return $requestData;
    }

    // }}}

    // {{{ Backend request

    protected function prepareRequestData($requestData, $type, $params)
    {
        $authorization = base64_encode(
            sprintf(
                '%s:%s',
                \Xlite\Core\Config::getInstance()->CDev->Paypal->loginClientId,
                \Xlite\Core\Config::getInstance()->CDev->Paypal->loginClientSecret
            )
        );
        $requestData['options']['headers']['Accept'] = 'application/json';
        $requestData['options']['headers']['Accept-Language'] = 'en_US';
        $requestData['options']['headers']['Content-Type'] = 'application/x-www-form-urlencoded';
        $requestData['options']['headers']['Authorization'] = sprintf('Basic %s', $authorization);

        return parent::prepareRequestData($requestData, $type, $params);
    }

    // }}}
}
