<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\PINCodes\Model\Repo;

use CDev\PINCodes\Model\PinCode as PinCodeModel;

class PinCode extends \XLite\Model\Repo\ARepo
{
    /**
     * Prepare certain search condition
     *
     * @param \Doctrine\ORM\QueryBuilder $queryBuilder Query builder to prepare
     * @param string                     $value        Condition data
     *
     * @return void
     */
    protected function prepareCndProduct(\Doctrine\ORM\QueryBuilder $queryBuilder, $value)
    {
        $queryBuilder
            ->andWhere('p.product=:product')
            ->setParameter('product', $value);
    }

    /**
     * Counts sold pin codes by product
     *
     * @param \XLite\Model\Product $product Product
     *
     * @return integer
     */
    public function countSold(\XLite\Model\Product $product)
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.product = :product AND p.isSold = :isSold')
            ->setParameter('isSold', true)
            ->setParameter('product', $product)
            ->count();
    }

    /**
     * Counts blocked pin codes by product
     *
     * @param \XLite\Model\Product $product Product
     *
     * @return integer
     */
    public function countBlocked(\XLite\Model\Product $product)
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.product = :product AND (p.isSold = :isSold OR p.isBlocked = :isBlocked)')
            ->setParameter('isSold', true)
            ->setParameter('isBlocked', true)
            ->setParameter('product', $product)
            ->count();
    }

    /**
     * Counts sold pin codes by product
     *
     * @param \XLite\Model\Product $product Product
     *
     * @return integer
     */
    public function countRemaining(\XLite\Model\Product $product)
    {
        return $this->createQueryBuilder('p')
            ->andWhere('p.product = :product AND p.isSold = :isSold AND p.isBlocked = :isBlocked')
            ->setParameter('isSold', false)
            ->setParameter('isBlocked', false)
            ->setParameter('product', $product)
            ->count();
    }

    protected function getUnavailablePinCodesIds()
    {
        $im = \XLite\Core\Database::getEM()->getUnitOfWork()->getIdentityMap();

        if (isset($im[$this->getEntityName()])) {
            return array_map(static function (PinCodeModel $pinCode) {
                return $pinCode->getId();
            }, array_filter($im[$this->getEntityName()], static function (PinCodeModel $pinCode) {
                return $pinCode->getIsSold() || $pinCode->getIsBlocked();
            }));
        }

        return [];
    }

    /**
     * Returns not sold pin code
     *
     * @param \XLite\Model\Product $product Product
     * @param integer              $count   Count
     *
     * @return PinCodeModel[]
     */
    public function getAvailablePinCodes(\XLite\Model\Product $product, $count)
    {
        $qb = $this->createQueryBuilder('p');

        $qb->andWhere('p.product = :product AND p.isSold = :isSold AND p.isBlocked = :isBlocked')
            ->addOrderBy('p.id')
            ->setParameter('isSold', false)
            ->setParameter('isBlocked', false)
            ->setParameter('product', $product)
            ->setMaxResults($count);

        if ($ids = $this->getUnavailablePinCodesIds()) {
            $qb->andWhere($qb->expr()->notIn(
                'p.id',
                $ids
            ));
        }

        return $qb->getResult();
    }
}
