<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\PINCodes\Model;

use Doctrine\ORM\Mapping as ORM;
use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 */
class Product extends \XLite\Model\Product
{
    /**
     * Whether pin codes are enabled for this product
     *
     * @var boolean
     *
     * @ORM\Column (type="boolean")
     */
    protected $pinCodesEnabled = false;

    /**
     * Whether to create pin codes automatically
     *
     * @var boolean
     *
     * @ORM\Column (type="boolean")
     */
    protected $autoPinCodes = false;

    /**
     * Pin codes (relation)
     *
     * @var \Doctrine\Common\Collections\ArrayCollection
     *
     * @ORM\OneToMany (targetEntity="CDev\PINCodes\Model\PinCode", mappedBy="product")
     */
    protected $pinCodes;

    /**
     * Constructor
     *
     * @param array $data Entity properties OPTIONAL
     *
     * @return void
     */
    public function __construct(array $data = [])
    {
        $this->pinCodes = new \Doctrine\Common\Collections\ArrayCollection();

        parent::__construct($data);
    }

    /**
     * Returns true if product has pin codes enabled and pin code autogeneration is turned off
     *
     * @return boolean
     */
    public function hasManualPinCodes()
    {
        return $this->getPinCodesEnabled() && !$this->getAutoPinCodes();
    }

    /**
     * Returns sold pins count
     *
     * @return integer
     */
    public function getSoldPinCodesCount()
    {
        return \XLite\Core\Database::getRepo('CDev\PINCodes\Model\PinCode')->countSold($this);
    }

    /**
     * Returns remaining pins count
     *
     * @return integer
     */
    public function getRemainingPinCodesCount()
    {
        return \XLite\Core\Database::getRepo('CDev\PINCodes\Model\PinCode')->countRemaining($this);
    }

    /**
     * @return int
     */
    public function getAmount()
    {
        return $this->pinCodesEnabled
            ? $this->getRemainingPinCodesCount()
            : parent::getAmount();
    }

    /**
     * Set pinCodesEnabled
     *
     * @param boolean $pinCodesEnabled
     * @return Product
     */
    public function setPinCodesEnabled($pinCodesEnabled)
    {
        $this->pinCodesEnabled = $pinCodesEnabled;
        return $this;
    }

    /**
     * Get pinCodesEnabled
     *
     * @return boolean
     */
    public function getPinCodesEnabled()
    {
        return $this->pinCodesEnabled;
    }

    /**
     * Set autoPinCodes
     *
     * @param boolean $autoPinCodes
     * @return Product
     */
    public function setAutoPinCodes($autoPinCodes)
    {
        $this->autoPinCodes = $autoPinCodes;
        return $this;
    }

    /**
     * Get autoPinCodes
     *
     * @return boolean
     */
    public function getAutoPinCodes()
    {
        return $this->autoPinCodes;
    }

    /**
     * Add pinCodes
     *
     * @param \CDev\PINCodes\Model\PinCode $pinCodes
     * @return Product
     */
    public function addPinCodes(\CDev\PINCodes\Model\PinCode $pinCodes)
    {
        $this->pinCodes[] = $pinCodes;
        return $this;
    }

    /**
     * Get pinCodes
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getPinCodes()
    {
        return $this->pinCodes;
    }
}
