<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\PINCodes\Model;

use ApiPlatform\Core\Annotation as ApiPlatform;
use Doctrine\ORM\Mapping as ORM;
use CDev\PINCodes\API\Endpoint\ProductPINCode\DTO\ProductPINCodeInput as Input;
use CDev\PINCodes\API\Endpoint\ProductPINCode\DTO\ProductPINCodeOutput as Output;

/**
 * PIN Code
 *
 * @ORM\Entity
 * @ORM\Table  (
 *      name="pin_codes",
 *      uniqueConstraints={
 *          @ORM\UniqueConstraint (name="productCode", columns={"code", "productId"})
 *      }
 * )
 * @ApiPlatform\ApiResource(
 *     shortName="Product PIN Code",
 *     input=Input::class,
 *     output=Output::class,
 *     itemOperations={
 *          "get"={
 *              "method"="GET",
 *              "path"="/products/{product_id}/pin_codes/{id}.{_format}",
 *              "identifiers"={"product_id", "id"},
 *              "requirements"={"product_id"="\d+", "id"="\d+"},
 *              "openapi_context"={
 *                  "parameters"={
 *                      {"name"="product_id", "in"="path", "required"=true, "schema"={"type"="integer"}},
 *                      {"name"="id", "in"="path", "required"=true, "schema"={"type"="integer"}}
 *                  }
 *              }
 *          },
 *          "put"={
 *              "method"="PUT",
 *              "path"="/products/{product_id}/pin_codes/{id}.{_format}",
 *              "identifiers"={"product_id", "id"},
 *              "requirements"={"product_id"="\d+", "id"="\d+"},
 *              "openapi_context"={
 *                  "parameters"={
 *                      {"name"="product_id", "in"="path", "required"=true, "schema"={"type"="integer"}},
 *                      {"name"="id", "in"="path", "required"=true, "schema"={"type"="integer"}}
 *                  }
 *              }
 *          },
 *          "delete"={
 *              "method"="DELETE",
 *              "path"="/products/{product_id}/pin_codes/{id}.{_format}",
 *              "identifiers"={"product_id", "id"},
 *              "requirements"={"product_id"="\d+", "id"="\d+"},
 *              "openapi_context"={
 *                  "parameters"={
 *                      {"name"="product_id", "in"="path", "required"=true, "schema"={"type"="integer"}},
 *                      {"name"="id", "in"="path", "required"=true, "schema"={"type"="integer"}}
 *                  }
 *              }
 *          }
 *     },
 *     collectionOperations={
 *          "get"={
 *              "method"="GET",
 *              "path"="/products/{product_id}/pin_codes.{_format}",
 *              "identifiers"={"product_id"},
 *              "requirements"={"product_id"="\d+"},
 *              "openapi_context"={
 *                  "parameters"={
 *                      {"name"="product_id", "in"="path", "required"=true, "schema"={"type"="integer"}}
 *                  },
 *              }
 *          },
 *          "post"={
 *              "method"="POST",
 *              "path"="/products/{product_id}/pin_codes.{_format}",
 *              "controller"="xcart.api.xc.pin_codes.product_pin_code.controller",
 *              "identifiers"={"product_id"},
 *              "requirements"={"product_id"="\d+"},
 *              "openapi_context"={
 *                  "parameters"={
 *                      {"name"="product_id", "in"="path", "required"=true, "schema"={"type"="integer"}}
 *                  }
 *              }
 *          }
 *     }
 * )
 */
class PinCode extends \XLite\Model\AEntity
{
    /**
     * PIN Code unique id
     *
     * @var integer
     *
     * @ORM\Id
     * @ORM\GeneratedValue (strategy="AUTO")
     * @ORM\Column         (type="integer")
     */
    protected $id;

    /**
     * Code
     *
     * @var string
     *
     * @ORM\Column (type="string", length=64)
     */
    protected $code = '';

    /**
     * Create date
     *
     * @var integer
     *
     * @ORM\Column (type="integer")
     */
    protected $createDate;

    /**
     * Is sold
     *
     * @var boolean
     *
     * @ORM\Column (type="boolean")
     */
    protected $isSold = false;

    /**
     * Is blocked
     *
     * @var boolean
     *
     * @ORM\Column (type="boolean")
     */
    protected $isBlocked = false;

    /**
     * Product (relation)
     *
     * @var \XLite\Model\Product
     *
     * @ORM\ManyToOne  (targetEntity="XLite\Model\Product", inversedBy="pinCodes")
     * @ORM\JoinColumn (name="productId", referencedColumnName="product_id", onDelete="SET NULL")
     */
    protected $product;

    /**
     * OrderItem (relation)
     *
     * @var \XLite\Model\OrderItem
     *
     * @ORM\ManyToOne  (targetEntity="XLite\Model\OrderItem", inversedBy="pinCodes")
     * @ORM\JoinColumn (name="orderItemId", referencedColumnName="item_id", onDelete="SET NULL")
     */
    protected $orderItem;


    /**
     * Generate pin code
     *
     * @return string
     *
     * @throws \Exception on attempt to autogenerate without $product defined
     */
    public function generateCode()
    {
        if (!$this->getProduct()) {
            throw new \Exception('Can not ensure pin uniqueness without a product assigned to this pin code');
        }

        $newValue = null;
        $repo = \XLite\Core\Database::getRepo('CDev\PINCodes\Model\PinCode');
        while (!$newValue || $repo->findOneBy(['code' => $newValue, 'product' => $this->getProduct()->getId()])) {
            $newValue = $this->getRandomCode();
        }
        $this->code = $newValue;

        return $this->code;
    }

    /**
     * Generates random pin code
     *
     * @return string
     */
    protected function getRandomCode()
    {
        return sprintf('%04d%04d%04d%04d', mt_rand(0, 9999), mt_rand(0, 9999), mt_rand(0, 9999), mt_rand(0, 9999));
    }


    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set code
     *
     * @param string $code
     * @return PinCode
     */
    public function setCode($code)
    {
        $this->code = $code;
        return $this;
    }

    /**
     * Get code
     *
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * Set createDate
     *
     * @param integer $createDate
     * @return PinCode
     */
    public function setCreateDate($createDate)
    {
        $this->createDate = $createDate;
        return $this;
    }

    /**
     * Get createDate
     *
     * @return integer
     */
    public function getCreateDate()
    {
        return $this->createDate;
    }

    /**
     * Set isSold
     *
     * @param boolean $isSold
     * @return PinCode
     */
    public function setIsSold($isSold)
    {
        $this->isSold = $isSold;
        return $this;
    }

    /**
     * Get isSold
     *
     * @return boolean
     */
    public function getIsSold()
    {
        return $this->isSold;
    }

    /**
     * Set isBlocked
     *
     * @param boolean $isBlocked
     * @return PinCode
     */
    public function setIsBlocked($isBlocked)
    {
        $this->isBlocked = $isBlocked;
        return $this;
    }

    /**
     * Get isBlocked
     *
     * @return boolean
     */
    public function getIsBlocked()
    {
        return $this->isBlocked;
    }

    /**
     * Set product
     *
     * @param \XLite\Model\Product $product
     * @return PinCode
     */
    public function setProduct(\XLite\Model\Product $product = null)
    {
        $this->product = $product;
        return $this;
    }

    /**
     * Get product
     *
     * @return \XLite\Model\Product
     */
    public function getProduct()
    {
        return $this->product;
    }

    /**
     * Set orderItem
     *
     * @param \XLite\Model\OrderItem $orderItem
     * @return PinCode
     */
    public function setOrderItem(\XLite\Model\OrderItem $orderItem = null)
    {
        $this->orderItem = $orderItem;
        return $this;
    }

    /**
     * Get orderItem
     *
     * @return \XLite\Model\OrderItem
     */
    public function getOrderItem()
    {
        return $this->orderItem;
    }
}
