<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\PINCodes\Logic\Import\Processor;

use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 */
abstract class Products extends \XLite\Logic\Import\Processor\Products
{
    /**
     * Get messages
     *
     * @return array
     */
    public static function getMessages()
    {
        return parent::getMessages()
               + [
                'PRODUCT-MANUAL-PIN-FMT' => 'Inventory tracking for product X will not be imported',
               ];
    }

    /**
     * Returns true if product has pin codes enabled and pin code autogeneration is turned off
     *
     * @return boolean
     */
    protected function hasManualPinCodes($sku, $column)
    {
        if (is_null($this->hasManualPinCodes)) {
            $result = false;

            $product = \XLite\Core\Database::getRepo('XLite\Model\Product')->findOneBy(
                [
                    'sku' => $sku
                ]
            );

            if ($product && $product->hasManualPinCodes()) {
                $result = true;
                $this->addWarning('PRODUCT-MANUAL-PIN-FMT', ['column' => $column, 'value' => $sku]);
            }

            $this->hasManualPinCodes = $result;
        }

        return $this->hasManualPinCodes;
    }

    /**
     * Verify data chunk
     *
     * @param array $data Data chunk
     *
     * @return boolean
     */
    protected function verifyData(array $data)
    {
        $this->hasManualPinCodes = null;

        return parent::verifyData($data);
    }

    /**
     * Import 'inventory tracking enabled' value
     *
     * @param \XLite\Model\Product $model  Product
     * @param mixed                $value  Value
     * @param array                $column Column info
     *
     * @return void
     */
    protected function importInventoryTrackingEnabledColumn(\XLite\Model\Product $model, $value, array $column)
    {
        if (!$this->hasManualPinCodes($model->getSku(), $column)) {
            parent::importInventoryTrackingEnabledColumn($model, $value, $column);
        }
    }

    /**
     * Import 'stock level' value
     *
     * @param \XLite\Model\Product $model  Product
     * @param mixed                $value  Value
     * @param array                $column Column info
     *
     * @return void
     */
    protected function importStockLevelColumn(\XLite\Model\Product $model, $value, array $column)
    {
        if (!$this->hasManualPinCodes($model->getSku(), $column)) {
            parent::importStockLevelColumn($model, $value, $column);
        }
    }
}
