<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace CDev\PINCodes\API\Endpoint\ProductPINCode\Transformer;

use ApiPlatform\Core\DataTransformer\DataTransformerInitializerInterface;
use ApiPlatform\Core\Exception\InvalidArgumentException;
use ApiPlatform\Core\Serializer\AbstractItemNormalizer;
use Doctrine\ORM\EntityManagerInterface;
use CDev\PINCodes\API\Endpoint\ProductPINCode\DTO\ProductPINCodeInput as InputDTO;
use CDev\PINCodes\Model\PinCode as Model;
use CDev\PINCodes\Model\Repo\PinCode as ModelRepo;
use XLite\Model\OrderItem;
use XLite\Model\Repo\OrderItem as OrderItemRepo;

class InputTransformer implements DataTransformerInitializerInterface, InputTransformerInterface
{
    protected EntityManagerInterface $entityManager;

    public function __construct(
        EntityManagerInterface $entityManager
    ) {
        $this->entityManager = $entityManager;
    }

    /**
     * @param InputDTO $object
     */
    public function transform($object, string $to, array $context = []): Model
    {
        /** @var Model $entity */
        $entity = $context[AbstractItemNormalizer::OBJECT_TO_POPULATE] ?? new Model();

        $productId = $this->detectProductId($context);
        if (!$productId) {
            throw new InvalidArgumentException('Product ID is invalid');
        }

        $this->checkUniqueness($entity, $object, $productId);

        $entity->setCode($object->code);
        $entity->setCreateDate($object->create_date ? $object->create_date->getTimestamp() : time());
        $entity->setIsSold($object->is_sold);
        $entity->setIsBlocked($object->is_blocked);

        /** @var OrderItem|null $orderItem */
        $orderItem = null;
        if ($object->order_item_id) {
            $orderItem = $this->getOrderItemRepository()->find($object->order_item_id);
            if (!$orderItem) {
                throw new InvalidArgumentException(sprintf('Order item with ID %d not found', $object->order_item_id));
            }
        }
        $entity->setOrderItem($orderItem);

        return $entity;
    }

    public function supportsTransformation($data, string $to, array $context = []): bool
    {
        if ($data instanceof Model) {
            return false;
        }

        return $to === Model::class && $context['input']['class'] === InputDTO::class;
    }

    /**
     * @return InputDTO
     */
    public function initialize(string $inputClass, array $context = [])
    {
        /** @var Model $entity */
        $entity = $context[AbstractItemNormalizer::OBJECT_TO_POPULATE] ?? null;
        if (!$entity) {
            return new InputDTO();
        }

        $input = new InputDTO();
        $input->code = $entity->getCode();
        $input->create_date = new \DateTimeImmutable('@' . $entity->getCreateDate());
        $input->is_sold = $entity->getIsSold();
        $input->is_blocked = $entity->getIsBlocked();
        $input->order_item_id = $entity->getOrderItem() ? $entity->getOrderItem()->getItemId() : null;

        return $input;
    }

    protected function getOrderItemRepository(): OrderItemRepo
    {
        return $this->entityManager->getRepository(OrderItem::class);
    }

    protected function getPinCodeItemRepository(): ModelRepo
    {
        return $this->entityManager->getRepository(Model::class);
    }

    protected function detectProductId(array $context): ?int
    {
        if (preg_match('/products\/(\d+)\//S', $context['request_uri'], $match)) {
            return (int) $match[1];
        }

        return null;
    }

    protected function checkUniqueness(Model $entity, InputDTO $object, int $productId): void
    {
        if (!$this->needUniqueness($entity, $object)) {
            return;
        }

        $count = $this->getPinCodeItemRepository()->countBy(['product' => $productId, 'code' => $object->code]);
        if ($count > 0) {
            throw new InvalidArgumentException(
                sprintf('PIN code "%s" must be unique', $object->code)
            );
        }
    }

    protected function needUniqueness(Model $entity, InputDTO $object): bool
    {
        return !$entity->isPersistent()
            || $entity->getCode() !== $object->code;
    }
}
