<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon\View\Form;

use Amazon\PayWithAmazon\Main;
use XCart\Domain\ModuleManagerDomain;

class SignUpForm extends \XLite\View\Form\AForm
{
    protected ?ModuleManagerDomain $moduleManagerDomain;

    public function __construct(array $params = [])
    {
        $this->moduleManagerDomain = \XCart\Container::getContainer()?->get(ModuleManagerDomain::class);

        parent::__construct($params);
    }

    /**
     * Return form attributes
     *
     * @return array
     */
    protected function getFormAttributes()
    {
        return [
            'action' => $this->getSignUpLinkUrl(),
            'method' => 'POST',
            'target' => '_blank'
        ];
    }

    protected function getFormParams()
    {
        return $this->getSignUpLinkData();
    }

    /**
     * @return array
     */
    protected function getSignUpLinkData(): array
    {
        $keys = $this->generateKeys();
        $company = \XLite\Core\Config::getInstance()->Company;
        $domain = \XLite\Core\Converter::buildFullURL('', '', [], \XLite::getCustomerScript(), true);
        $privacyUrl = \XLite\Core\Converter::buildFullURL('privacy_policy', '', [], \XLite::getCustomerScript());
        $callbackUrl = \XLite\Core\Converter::buildFullURL('callback', '', [], \XLite::getCustomerScript());

        $module = $this->moduleManagerDomain->getModule('Amazon-PayWithAmazon');

        return [
            'spId'                  => Main::SP_IDS[$this->getMethod()->getSetting('region')],
            'source'                => 'SPPL',
            'locale'                => Main::getLocaleCode(),
            'publicKey'             => $this->getPublicKeyString($keys['publickey']),
            'onboardingVersion'     => 2,
            'spSoftwareVersion'     => \XLite::getInstance()->getVersion(),
            'spAmazonPluginVersion' => $module['version'],
            'keyShareURL'       => $this->buildFullURL(
                'pay_with_amazon',
                'register_callback',
                ['token' => $keys['token']]
            ),
            'merchantStoreDescription' => $company->company_name,
            'merchantLoginDomains[]'   => rtrim(strtok($domain, '?'), '/'),
            'merchantPrivacyNoticeURL' => $privacyUrl,
            'merchantSandboxIPNURL'    => $callbackUrl,
            'merchantProductionIPNURL' => $callbackUrl
        ];
    }

    /**
     * @return array
     */
    public function generateKeys()
    {
        $method = $this->getMethod();
        $tokenManager = new \Amazon\PayWithAmazon\Core\TokenManager();

        $rsa = new \phpseclib\Crypt\RSA();
        $rsa->setPrivateKeyFormat(\phpseclib\Crypt\RSA::PRIVATE_FORMAT_PKCS8);
        $keys = $rsa->createKey(2048);
        $keys['token'] = md5($keys['publickey']);
        $method->setSetting('token', $keys['token']);
        $method->setSetting('public_key', $keys['publickey']);
        $method->setSetting('private_key_generated', $tokenManager->encode($keys['privatekey']));

        $method->update();

        return $keys;
    }

    /**
     * @return string|null
     */
    public function getPublicKeyString(string $publicKey)
    {
        $trim = ['-----BEGIN PUBLIC KEY-----', '-----END PUBLIC KEY-----', "\n"];
        $publicKey = str_replace($trim, [''], $publicKey);
        $publicKey = preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $publicKey);

        return $publicKey;
    }

    /**
     * @return string
     */
    protected function getSignUpLinkUrl(): string
    {
        $subDomain = $this->getMethod()->getSetting('region') === 'USD' ? 'payments' : 'payments-eu';

        return 'https://' . $subDomain . '.amazon.com/register';
    }

    /**
     * @return \XLite\Model\Payment\Method
     */
    protected function getMethod()
    {
        return \Amazon\PayWithAmazon\Main::getMethod();
    }
}
