<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon\Model\Payment\Processor;

use Amazon\PayWithAmazon\Main;
use XLite\Core\TopMessage;
use XLite\InjectLoggerTrait;
use XLite\Model\Payment\BackendTransaction;
use XLite\Model\Payment\Transaction;

/**
 * PayWithAmazon processor
 */
class PayWithAmazon extends \XLite\Model\Payment\Base\CreditCard
{
    use InjectLoggerTrait;

    /**
     * @var array
     */
    protected $jsUrls = [
        'EUR' => 'https://static-eu.payments-amazon.com/checkout.js',
        'GBP' => 'https://static-eu.payments-amazon.com/checkout.js',
        'USD' => 'https://static-na.payments-amazon.com/checkout.js',
    ];

    /**
     * @var array
     */
    protected $ipnData = [];

    /**
     * Get allowed backend transactions
     * @todo: check for partial/multi refund
     *
     * @return string[] Status code
     */
    public function getAllowedTransactions()
    {
        return [
            BackendTransaction::TRAN_TYPE_CAPTURE,
            BackendTransaction::TRAN_TYPE_VOID,
            BackendTransaction::TRAN_TYPE_REFUND,
            BackendTransaction::TRAN_TYPE_REFUND_PART,
            BackendTransaction::TRAN_TYPE_REFUND_MULTI
        ];
    }

    /**
     * @return array
     */
    public static function getRedirectToCartReasons()
    {
        return [
            'AmazonRejected',
            'TransactionTimedOut',
            'ProcessingFailure',
            'MFA_Failure'
        ];
    }

    /**
     * @return string Widget class name or template path
     */
    public function getSettingsWidget()
    {
        return 'modules/Amazon/PayWithAmazon/config.twig';
    }

    /**
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string
     */
    public function getCheckoutTemplate(\XLite\Model\Payment\Method $method)
    {
        return 'modules/Amazon/PayWithAmazon/checkout/steps/shipping/parts/paymentMethod.twig';
    }

    /**
     * Return IPN endpoint URL
     *
     * @return string
     */
    public function getAmazonIPNURL()
    {
        return \XLite::getInstance()->getShopURL(
            \XLite\Core\Converter::buildFullURL('callback', '', [], \XLite::CART_SELF),
            \XLite\Core\Config::getInstance()->Security->customer_security
        );
    }

    /**
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string
     */
    public function getJsSdkUrl($method)
    {
        $currency = $method->getSetting('region');
        $currency = in_array($currency, ['EUR', 'GBP'], true)
            ? $currency
            : 'USD';

        return $this->jsUrls[$currency];
    }

    /**
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string|boolean|null
     */
    public function getAdminIconURL(\XLite\Model\Payment\Method $method)
    {
        return true;
    }

    /**
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return boolean
     */
    public function isConfigured(\XLite\Model\Payment\Method $method)
    {
        return parent::isConfigured($method)
        && $method->getSetting('merchant_id')
        && $method->getSetting('client_id')
        && \XLite\Core\Config::getInstance()->Security->customer_security;
    }

    /**
     * Get initial transaction type (used when customer places order)
     *
     * @param \XLite\Model\Payment\Method $method Payment method object OPTIONAL
     *
     * @return string
     */
    public function getInitialTransactionType($method = null)
    {
        return ($method ? $method->getSetting('capture_mode') : $this->getSetting('capture_mode')) === 'A'
            || ($method ? $method->getSetting('sync_mode') : $this->getSetting('sync_mode')) === 'A'
            ? BackendTransaction::TRAN_TYPE_AUTH
            : BackendTransaction::TRAN_TYPE_SALE;
    }

    /**
     * Get allowed currencies
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return array
     */
    protected function getAllowedCurrencies(\XLite\Model\Payment\Method $method)
    {
        return ['USD', 'GBP', 'EUR'];
    }

    /**
     * Do initial payment
     *
     * @return string Status code
     */
    protected function doInitialPayment()
    {
        $resultStatus = static::FAILED;

        $isAjax = \XLite\Core\Request::getInstance()->isAJAX();
        $sessionId = $this->transaction->getDetail('checkoutSessionId');
        if (!$sessionId) {
            \XLite\Core\Event::createAmazonCheckout(['type' => 'ProcessOrder']);

            if ($isAjax) {
                \XLite\Core\Event::getInstance()->display();
                \XLite\Core\Event::getInstance()->clear();
            }

            return static::SILENT;
        }

        $payload = $this->getPayloadData($this->transaction);
        $responseData = $this->getApiResponse('updateCheckoutSession', [$sessionId, $payload]);

        $redirectUrl = $responseData['webCheckoutDetails']['amazonPayRedirectUrl'] ?? null;
        if ($redirectUrl) {
            \XLite\Core\Operator::redirect($redirectUrl, $isAjax ? 200 : 302);
        } else {
            TopMessage::addWarning('Payment update error: {{error}}', ['error' => $responseData['error']]);
        }

        return $resultStatus;
    }

    /**
     * Note: placeholder for workflow without redirect via static::SILENT
     * @return array
     */
    public function getPaymentWidgetData()
    {
        return [];
    }

    /**
     * @param Transaction $transaction
     * @param bool $address OPTIONAL
     *
     * @return array
     */
    public function getPayloadData(Transaction $transaction, $address = false)
    {
        $this->transaction = $transaction;
        //logic described here ECOM-3484
        $isAuthorizeMode = $this->getSetting('capture_mode') === 'A' || $this->getSetting('sync_mode') === 'A';

        $result = [
            'paymentDetails' => [
                'paymentIntent' => $isAuthorizeMode ? 'Authorize' : 'AuthorizeWithCapture',
                'canHandlePendingAuthorization' => $isAuthorizeMode && $this->getSetting('sync_mode') === 'A',
                'chargeAmount' => [
                    'amount' => $this->transaction->getValue(),
                    'currencyCode' => $this->transaction->getCurrency()->getCode()
                ]
            ],
            'merchantMetadata' => [
                'merchantReferenceId' => $this->transaction->getPublicTxnId(),
                'merchantStoreName' => \XLite\Core\Config::getInstance()->Company->company_name,
                'noteToBuyer' => (string) static::t('Thank you for your order')
            ],
            'webCheckoutDetails' => [
                'checkoutResultReturnUrl' => $this->getReturnURL(null, true)
            ],
            'platformId' => Main::PLATFORM_IDS[$this->getSetting('region')]
        ];

        if (!$isAuthorizeMode) {
            $result['paymentDetails']['softDescriptor'] = $this->getStoreName();
        }

        if (
            $address
            && ($profile = $this->transaction->getProfile())
            && ($shippingAddress = $profile->getShippingAddress())
        ) {
            $result['addressDetails'] = $this->prepareAddress($shippingAddress);
            $result['paymentDetails']['presentmentCurrency'] = $this->transaction->getCurrency()->getCode();
        }

        return $result;
    }

    /**
     * @param \XLite\Model\Address $address
     *
     * @return array
     */
    protected function prepareAddress(\XLite\Model\Address $address)
    {
        $countryCode = $address->getCountryCode();
        $state = $address->getState();

        $result = [
            'name' => $address->getName(),
            'addressLine1' => $address->getAddress1(),
            'city' => $address->getCity(),
            'stateOrRegion' => $state,
            'postalCode' => $address->getZipcode(),
            'countryCode' => $countryCode,
            'phoneNumber' => $address->getPhone()
        ];

        if (in_array($countryCode, ['US', 'CA'], true) && $state) {
            $result['stateOrRegion'] = $state->getCode();
        }

        return $result;
    }

    /**
     * Process return
     *
     * @param Transaction $transaction Return-owner transaction
     *
     * @return void
     * @throws \Exception
     */
    public function processReturn(Transaction $transaction)
    {
        parent::processReturn($transaction);

        $transactionStatus = $transaction::STATUS_FAILED;

        $data = \XLite\Core\Request::getInstance()->getData();
        $sessionId = $data['amazonCheckoutSessionId'];
        $payload = [
            'chargeAmount' => [
                'amount' => $this->transaction->getValue(),
                'currencyCode' => $this->transaction->getCurrency()->getCode()
            ]
        ];
        $responseData = $this->getApiResponse('completeCheckoutSession', [$sessionId, $payload]);

        $status = $responseData['statusDetails']['state'] ?? null;

        if ($status === 'Completed') {
            $pending = $responseData['status_code'] === 202;
            $backendTransaction = $this->getBackendTransactionByType(
                $transaction,
                $this->getInitialTransactionType(),
                $this->transaction->getValue()
            );

            if ($backendTransaction) {
                $backendTransaction->setStatus($pending ? BackendTransaction::STATUS_PENDING : BackendTransaction::STATUS_SUCCESS);
            }

            $transactionStatus = $pending
                ? $transaction::STATUS_PENDING
                : $transaction::STATUS_SUCCESS;

            $this->transaction->setDataCell('chargeId', $responseData['chargeId'], 'chargeId', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
            $this->transaction->setDataCell('chargePermissionId', $responseData['chargePermissionId'], 'amazonChargePermissionId', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
        } else {
            $reason = $responseData['response']['reasonCode'] ?? null;

            switch ($reason) {
                case 'CheckoutSessionCanceled':
                    $transactionStatus = $transaction::STATUS_CANCELED;
                    break;
                case 'AmazonRejected':
                    $transactionStatus = $transaction::STATUS_VOID;
                    break;
            }

            TopMessage::addWarning('Payment complete error: {{error}}', ['error' => $responseData['error']]);
        }

        $this->transaction->setStatus($transactionStatus);
    }

    /**
     * @param array $responseAddress
     *
     * @return array
     */
    public function getAddressDataFromOrderReferenceDetails($responseAddress)
    {
        $address         = [];
        if (!empty($responseAddress)) {
            $address['zipcode']      = $responseAddress['postalCode'];
            $address['country_code'] = $responseAddress['countryCode'];
            $address['city']         = $responseAddress['city'];

            $state = \XLite\Core\Database::getRepo('XLite\Model\State')
                ->findOneByCountryAndState($address['country_code'], $responseAddress['stateOrRegion']);

            if ($state) {
                $address['state_id'] = $state->getStateId();
            } elseif (!empty($responseAddress['stateOrRegion'])) {
                $address['custom_state'] = $responseAddress['stateOrRegion'];
            }

            if (!empty($responseAddress['phoneNumber'])) {
                $address['phone'] = $responseAddress['phoneNumber'];
            }

            if (empty($responseAddress['addressLine1']) && !empty($responseAddress['addressLine2'])) {
                $responseAddress['addressLine1'] = $responseAddress['addressLine2'];
                unset($responseAddress['addressLine2']);
            }
            if (!empty($responseAddress['addressLine1'])) {
                $address['address1'] = $responseAddress['addressLine1'];
            }
            if (!empty($responseAddress['addressLine2'])) {
                $address['address2'] = $responseAddress['addressLine2'];
            }

            [$address['firstname'], $address['lastname']] = explode(' ', $responseAddress['name'], 2);
            if (empty($address['lastname'])) {
                // XC does not support single word customer name
                $address['lastname'] = $address['firstname'];
            }
        }

        return $address;
    }

    /**
     * @param Transaction $transaction
     *
     * @return boolean
     */
    protected function isCaptureTransactionAllowed(Transaction $transaction)
    {
        return $transaction->isCaptureTransactionAllowed()
            && ((bool) $this->getOrderChargeId($transaction));
    }

    /**
     * @param BackendTransaction $transaction Transaction
     *
     * @return boolean
     */
    protected function doCapture(BackendTransaction $transaction)
    {
        $result = false;
        $paymentTransaction = $transaction->getPaymentTransaction();

        $chargeId = $this->getOrderChargeId($paymentTransaction);

        $payload = [
            'captureAmount' => [
                'amount' => $transaction->getValue(),
                'currencyCode' => $paymentTransaction->getCurrency()->getCode()
            ],
            'softDescriptor' => $this->getStoreName()
        ];
        $headers = ['x-amz-pay-Idempotency-Key' => uniqid()];
        $responseData = $this->getApiResponse('captureCharge', [$chargeId, $payload, $headers]);

        $status = $responseData['statusDetails']['state'] ?? null;
        if ($status && $status !== 'Declined') {
            $transaction->setStatus(
                $status === 'Captured'
                    ? BackendTransaction::STATUS_SUCCESS
                    : BackendTransaction::STATUS_PENDING
            );

            $this->transaction->setDataCell('lastUpdatedTimestamp', $responseData['statusDetails']['lastUpdatedTimestamp'], '', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);

            if (!$paymentTransaction->getDetail('chargeId')) {
                $this->transaction->setDataCell('chargeId', $responseData['chargeId'], 'chargeId', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
                $this->transaction->setDataCell('chargePermissionId', $responseData['chargePermissionId'], 'amazonChargePermissionId', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
            }

            $result = true;
            TopMessage::addInfo('Payment has been captured successfully.');
        } else {
            TopMessage::addWarning('Payment capture error: {{error}}', ['error' => $responseData['error']]);
        }

        return $result;
    }

    /**
     * @param Transaction $transaction
     *
     * @return boolean
     */
    protected function isVoidTransactionAllowed(Transaction $transaction)
    {
        return $transaction->isVoidTransactionAllowed()
            && ((bool) $this->getOrderChargeId($transaction));
    }

    /**
     * @param BackendTransaction $transaction Transaction
     *
     * @return boolean
     */
    protected function doVoid(BackendTransaction $transaction)
    {
        $result = false;
        $paymentTransaction = $transaction->getPaymentTransaction();

        $payload = [
            'cancellationReason' => ''
        ];

        $chargeId = $this->getOrderChargeId($paymentTransaction);
        $responseData = $this->getApiResponse('cancelCharge', [$chargeId, $payload]);

        $status = $responseData['statusDetails']['state'] ?? null;
        if ($status === 'Canceled') {
            $transaction->setStatus(BackendTransaction::STATUS_SUCCESS);
            $paymentTransaction->setStatus(Transaction::STATUS_VOID);

            $result = true;
            TopMessage::addInfo('Payment have been voided successfully.');
        } else {
            TopMessage::addWarning('Payment void error: {{error}}', ['error' => $responseData['error']]);
        }

        return $result;
    }

    /**
     * @param Transaction $transaction
     *
     * @return boolean
     */
    protected function isRefundTransactionAllowed(Transaction $transaction)
    {
        return $transaction->isRefundTransactionAllowed()
        && (bool) $this->getOrderCaptureId($transaction)
        && !(bool) $transaction->getDetail('refundId');
    }

    /**
     * @param BackendTransaction $transaction Transaction
     *
     * @return boolean
     */
    protected function doRefund(BackendTransaction $transaction)
    {
        $result = false;
        $paymentTransaction = $transaction->getPaymentTransaction();

        $payload = [
            'chargeId' => $this->getOrderCaptureId($paymentTransaction),
            'refundAmount' => [
                'amount' => $transaction->getValue(),
                'currencyCode' => $paymentTransaction->getCurrency()->getCode()
            ]
        ];
        $headers = ['x-amz-pay-Idempotency-Key' => uniqid()];
        $responseData = $this->getApiResponse('createRefund', [$payload, $headers]);

        $status = $responseData['statusDetails']['state'] ?? null;
        if ($status && $status !== 'Declined') {
            $transaction->setStatus(
                $status === 'Refunded'
                    ? BackendTransaction::STATUS_SUCCESS
                    : BackendTransaction::STATUS_PENDING
            );

            $paymentTransaction->setDataCell('refundId', $responseData['refundId'], 'AmazonRefundId identifier');

            $result = true;
            TopMessage::addInfo(
                $status === 'Refunded'
                    ? 'Payment has been refunded successfully.'
                    : 'Refund is in progress...'
            );
        } else {
            TopMessage::addWarning('Payment refund error: {{error}}', ['error' => $responseData['error']]);
        }

        return $result;
    }

    /**
     * @param BackendTransaction $transaction Transaction
     *
     * @return boolean
     */
    protected function doRefundPart(BackendTransaction $transaction)
    {
        return $this->doRefund($transaction);
    }

    /**
     * @param BackendTransaction $transaction Transaction
     *
     * @return boolean
     */
    protected function doRefundMulti(BackendTransaction $transaction)
    {
        return $this->doRefund($transaction);
    }

    /**
     * @param Transaction $transaction
     *
     * @return mixed
     */
    protected function getOrderChargeId(Transaction $transaction)
    {
        $chargeId = $transaction->getDetail('chargeId');
        if (!$chargeId) {
            $authId = $transaction->getDetail('amazonAuthorizationId');
            $chargeId = $authId ? (substr_replace($authId, 'C', 20, 1)) : null;
        }

        return $chargeId;
    }

    /**
     * @param Transaction $transaction
     *
     * @return mixed
     */
    protected function getOrderCaptureId(Transaction $transaction)
    {
        $chargeId = $transaction->getDetail('chargeId');

        return $chargeId ?? $transaction->getDetail('amazonCaptureId');
    }

    /**
     * Get callback request owner transaction or null
     *
     * @return Transaction|null
     */
    public function getCallbackOwnerTransaction()
    {
        $result = null;

        $headers     = getallheaders();
        $requestBody = file_get_contents('php://input');

        $ipnData = json_decode($requestBody, true);
        $messageData = isset($ipnData['Message']) ? json_decode($ipnData['Message'], true) : null;

        $this->getLogger('Amazon-PayWithAmazon')->debug(__FUNCTION__, [
            'headers' => $headers,
            'request' => $requestBody,
            'data' => $messageData,
        ]);

        if (isset($messageData['ObjectType'])) {
            $method = null;
            switch ($messageData['ObjectType']) {
                case 'CHARGE':
                    $method = 'getCharge';
                    break;
                case 'REFUND':
                    $method = 'getRefund';
                    break;
            }

            if ($method) {
                $responseData = $this->getApiResponse($method, [$messageData['ObjectId']]);
                if (!isset($responseData['error'])) {
                    $chargeId = $responseData['chargeId'] ?? null;
                    $chargePermissionId = $responseData['chargePermissionId'] ?? null;
                    if ($chargeId) {
                        $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                            'chargeId',
                            $chargeId
                        );
                    } elseif ($chargePermissionId) {
                        $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                            'chargePermissionId',
                            $chargePermissionId
                        );
                    }

                    $responseData['ObjectType'] = $messageData['ObjectType'];
                    $this->ipnData = $responseData;
                } else {
                    TopMessage::addWarning('Payment refund error: {{error}}', ['error' => $responseData['error']]);
                }
            }
        }

        if (isset($messageData['Version'])) {
            $xmlData = simplexml_load_string($messageData['NotificationData']);
            $xmlJson = json_encode($xmlData);
            $notificationData = json_decode($xmlJson, true);

            $authorizationId = $notificationData['AuthorizationDetails']['AmazonAuthorizationId'] ?? null;
            $chargeId = $notificationData['CaptureDetails']['AmazonCaptureId'] ?? null;
            $refundId = $notificationData['RefundDetails']['AmazonRefundId'] ?? null;
            if ($authorizationId) {
                $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                    'amazonAuthorizationId',
                    $authorizationId
                );
            } elseif ($chargeId) {
                $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                    'chargeId',
                    $chargeId
                );
                if (!$result) {
                    $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                        'amazonCaptureId',
                        $chargeId
                    );
                }
            } elseif ($refundId) {
                $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                    'refundId',
                    $refundId
                );
                if (!$result) {
                    $result = \Xlite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByCell(
                        'amazonRefundId',
                        $refundId
                    );
                }
            }

            $this->ipnData = $messageData;
            $this->ipnData['NotificationData'] = $notificationData;
        }

        return $result;
    }

    /**
     * @param Transaction $transaction Callback-owner transaction
     */
    public function processCallback(Transaction $transaction)
    {
        parent::processCallback($transaction);

        $ipnData = $this->ipnData;

        if (!$ipnData || !$this->canProcessCallback($transaction)) {
            return;
        }

        if (isset($ipnData['Version'])) {
            $this->getLogger('Amazon-PayWithAmazon')->debug('payment gateway : v1 callback process', $ipnData);

            switch ($ipnData['NotificationType']) {
                case 'PaymentCapture':
                    $status = $ipnData['NotificationData']['CaptureDetails']['CaptureStatus']['State'] ?? null;
                    switch ($status) {
                        case 'Completed':
                            $resultTransaction = $this->getBackendTransactionByType(
                                $transaction,
                                BackendTransaction::TRAN_TYPE_CAPTURE,
                                $this->transaction->getValue()
                            );

                            if ($resultTransaction) {
                                $resultTransaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                            }

                            break;
                        case 'Declined':
                            $resultTransaction = $this->getBackendTransactionByType(
                                $transaction,
                                BackendTransaction::TRAN_TYPE_AUTH,
                                $this->transaction->getValue()
                            );

                            if ($resultTransaction) {
                                $resultTransaction->setStatus(BackendTransaction::STATUS_FAILED);
                                $transaction->setStatus(BackendTransaction::STATUS_FAILED);
                            }

                            break;
                    }

                    break;
                case 'PaymentRefund':
                    $status = $ipnData['NotificationData']['RefundDetails']['RefundStatus']['State'] ?? null;
                    switch ($status) {
                        case 'Completed':
                            $resultTransaction = $this->getBackendTransactionByType(
                                $transaction,
                                BackendTransaction::TRAN_TYPE_REFUND_MULTI,
                                $this->transaction->getValue()
                            );

                            if ($resultTransaction) {
                                $resultTransaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                            }

                            break;
                    }

                    break;
            }
        }

        $lastUpdated = $transaction->getDataCell('lastUpdatedTimestamp');
        $ipnTimestamp = $ipnData['statusDetails']['lastUpdatedTimestamp'];
        if ($lastUpdated && $lastUpdated->getValue() === $ipnTimestamp) {
            return; //additional prevent double requests cuz amazon sucks
        }

        $transaction->setDataCell('lastUpdatedTimestamp', $ipnTimestamp, '', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
        $transaction->setEntityLock(\XLite\Model\Payment\Transaction::LOCK_TYPE_IPN, 30);

        $this->getLogger('Amazon-PayWithAmazon')->debug('payment gateway : callback', $ipnData);

        $resultTransaction = null;
        $status = $ipnData['statusDetails']['state'];
        $currency = $transaction->getCurrency();

        switch ($ipnData['ObjectType']) {
            case 'CHARGE':
                switch ($status) {
                    case 'Authorized':
                        $resultTransaction = $this->getBackendTransactionByType(
                            $transaction,
                            BackendTransaction::TRAN_TYPE_AUTH,
                            $ipnData['chargeAmount']['amount']
                        );

                        if ($resultTransaction) {
                            $resultTransaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                            $transaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                        }

                        $isCaptureOnAsync = $this->getSetting('capture_mode') === 'C' && $this->getSetting('sync_mode') === 'A';
                        if ($isCaptureOnAsync) {
                            $captureTransaction = $this->getBackendTransactionByType(
                                $transaction,
                                BackendTransaction::TRAN_TYPE_CAPTURE,
                                $ipnData['chargeAmount']['amount']
                            );

                            $this->doCapture($captureTransaction);

                            $captureTransaction->registerTransactionInOrderHistory('callback, IPN');
                        }

                        break;
                    case 'Captured':
                        $resultTransaction = $this->getBackendTransactionByType(
                            $transaction,
                            BackendTransaction::TRAN_TYPE_CAPTURE,
                            $ipnData['captureAmount']['amount']
                        );

                        if ($resultTransaction) {
                            $resultTransaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                        }

                        break;
                    case 'Canceled':
                        $amount = $currency->roundValueAsInteger($transaction->getValue());

                        $resultTransaction = $this->getBackendTransactionByType(
                            $transaction,
                            BackendTransaction::TRAN_TYPE_VOID,
                            $amount
                        );

                        if ($resultTransaction) {
                            $resultTransaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                            $transaction->setStatus(Transaction::STATUS_VOID);
                        }

                        break;
                    case 'Declined':
                        $resultTransaction = $this->getBackendTransactionByType(
                            $transaction,
                            BackendTransaction::TRAN_TYPE_AUTH,
                            $ipnData['chargeAmount']['amount']
                        );

                        if ($resultTransaction) {
                            $resultTransaction->setStatus(BackendTransaction::STATUS_FAILED);
                            $transaction->setStatus(BackendTransaction::STATUS_FAILED);
                        }

                        break;
                }

                break;

            case 'REFUND':
                switch ($status) {
                    case 'Refunded':
                        $resultTransaction = $this->getBackendTransactionByType(
                            $transaction,
                            BackendTransaction::TRAN_TYPE_REFUND_MULTI,
                            $ipnData['refundAmount']['amount']
                        );

                        if ($resultTransaction) {
                            $resultTransaction->setStatus(BackendTransaction::STATUS_SUCCESS);
                        }

                        break;
                    case 'Declined':
                        $resultTransaction = $this->getBackendTransactionByType(
                            $transaction,
                            BackendTransaction::TRAN_TYPE_REFUND,
                            $ipnData['refundAmount']['amount']
                        );

                        if ($resultTransaction) {
                            $resultTransaction->setStatus(BackendTransaction::STATUS_FAILED);
                        }

                        break;
                }

                break;
        }

        if ($resultTransaction) {
            $resultTransaction->registerTransactionInOrderHistory('callback, IPN');
        } else {
            $transaction->registerTransactionInOrderHistory('callback, IPN');
        }

        if ($transaction->isEntityLocked(\XLite\Model\Payment\Transaction::LOCK_TYPE_IPN)) {
            $transaction->unsetEntityLock(\XLite\Model\Payment\Transaction::LOCK_TYPE_IPN);
        }
    }

    /**
     * @param Transaction $transaction
     * @param string      $type
     * @param int|float   $amount
     *
     * @return null|BackendTransaction
     */
    protected function getBackendTransactionByType(Transaction $transaction, $type, $amount)
    {
        $value = null;

        $resultTransaction = $this->findLastBackendTransactionByType($transaction, $type);
        if ($resultTransaction) {
            $value = $resultTransaction->getValue();
        }

        if ($value !== (float) $amount) {
            $type = $amount < $transaction->getValue() ? $this->getPartialTransactionType($type) : $type;
            $resultTransaction = $transaction->createBackendTransaction($type);
            $resultTransaction->setValue($amount);

            if (
                in_array($type, [
                    BackendTransaction::TRAN_TYPE_CAPTURE_PART,
                    BackendTransaction::TRAN_TYPE_REFUND_PART
                ])
            ) {
                $transaction->setValue($amount);
            }
        }

        if ($resultTransaction) {
            $transaction->setType($type);
        }

        return $resultTransaction;
    }

    /**
     * @param string $type
     *
     * @return string
     */
    protected function getPartialTransactionType($type)
    {
        $result = $type;

        switch ($type) {
            case BackendTransaction::TRAN_TYPE_CAPTURE:
                $result = BackendTransaction::TRAN_TYPE_CAPTURE_PART;
                break;
            case BackendTransaction::TRAN_TYPE_REFUND:
                $result = BackendTransaction::TRAN_TYPE_REFUND_PART;
                break;
        }

        return $result;
    }

    /**
     * @param Transaction $transaction
     * @param string      $type
     *
     * @return null|BackendTransaction
     */
    protected function findLastBackendTransactionByType(Transaction $transaction, $type)
    {
        /** @var BackendTransaction[] $backendTransactions */
        $backendTransactions = $transaction->getBackendTransactions()->toArray();

        usort($backendTransactions, static function ($a, $b) {
            $aId = $a->getId();
            $bId = $b->getId();

            if ($aId === $bId) {
                return 0;
            }

            return $aId > $bId ? -1 : 1;
        });

        foreach ($backendTransactions as $backendTransaction) {
            if (
                $backendTransaction->getType() === $type
                && $backendTransaction->getStatus() !== $backendTransaction::STATUS_SUCCESS
            ) {
                return $backendTransaction;
            }
        }

        return null;
    }

    /**
     * @param \XLite\Model\Payment\Transaction $transaction
     *
     * @return boolean
     */
    protected function canProcessCallback(\XLite\Model\Payment\Transaction $transaction)
    {
        return !$this->isIPNLocked($transaction) && $this->isOrderProcessed($transaction);
    }

    /**
     * @param \XLite\Model\Payment\Transaction $transaction
     *
     * @return bool
     */
    protected function isIPNLocked(\XLite\Model\Payment\Transaction $transaction)
    {
        return $transaction->isEntityLocked(\XLite\Model\Payment\Transaction::LOCK_TYPE_IPN)
            && !$transaction->isEntityLockExpired(\XLite\Model\Payment\Transaction::LOCK_TYPE_IPN);
    }

    /**
     * @param \XLite\Model\Payment\Transaction $transaction
     *
     * @return bool
     */
    protected function isOrderProcessed(\XLite\Model\Payment\Transaction $transaction)
    {
        return !$transaction->isOpen() && !$transaction->isInProgress() && $transaction->getOrder()->getOrderNumber();
    }

    protected function sendUpdatePaymentInfoMail(Transaction $transaction)
    {
        $order = $transaction->getOrder();
        $order->setIsNotificationsAllowedFlag(false);

        \XLite\Core\Mailer::sendUpdateAmazonPaymentInfo($order);
    }

    /**
     * Define saved into transaction data schema
     *
     * @return array
     */
    protected function defineSavedData()
    {
        $data = parent::defineSavedData();

        $data['amazonOrderReferenceId'] = 'Amazon order reference ID';
        $data['authorizationStatus']    = 'Current status of the authorization';
        $data['authorizationReason']    = 'Current reason of the authorization';
        $data['amazonAuthorizationId']  = 'The Amazon-generated identifier for this authorization transaction';
        $data['amazonCaptureId']        = 'AmazonCaptureId identifier';

        return $data;
    }

    /**
     * @return \XLite\Model\Payment\Transaction|null
     */
    public function getReturnOwnerTransaction()
    {
        $txn = null;
        $txnIdName = \XLite\Model\Payment\Base\Online::RETURN_TXN_ID;

        if (!empty(\XLite\Core\Request::getInstance()->$txnIdName)) {
            $txn = \XLite\Core\Database::getRepo('XLite\Model\Payment\Transaction')
                ->findOneByPublicTxnId(\XLite\Core\Request::getInstance()->$txnIdName);
        }

        if ($txn) {
            if ($retryTxnId = $txn->getDetail('retry_txn_id')) {
                $txn = \XLite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByPublicTxnId($retryTxnId) ?: $txn;
            }
        }

        return $txn;
    }

    protected function getApiResponse(string $method, $params)
    {
        try {
            $client = new \Amazon\Pay\API\Client($this->getAuthData());
            $result = call_user_func_array([$client, $method], $params);

            $resultData = json_decode($result['response'], true);

            if (in_array($result['status'], [200, 201, 202])) {
                $resultData['status_code'] = $result['status'];
                $response = $resultData;
            } else {
                $response = [
                    'error' => $resultData['message'] ?? $method,
                    'method' => $method,
                    'status' => $result['status'],
                    'response' => $resultData
                ];

                $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
            }
        } catch (\Exception $e) {
            $response = [
                'error' => $e->getMessage(),
                'method' => $method,
            ];

            $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
        }

        return $response;
    }

    protected function getAuthData(): array
    {
        $tokenManager = new \Amazon\PayWithAmazon\Core\TokenManager();

        return \Amazon\PayWithAmazon\Main::getAuthData($tokenManager);
    }

    /**
     * @param \XLite\Model\Order $order Order
     *
     * @return string
     */
    public function getPaymentDescription($order)
    {
        $transaction = $order->getPaymentTransactions()->last();
        $sessionId = $transaction ? $transaction->getDataCell('checkoutSessionId') : null;
        $paymentMethodName = $sessionId ? $transaction->getDataCell('paymentMethodName') : null;

        return $paymentMethodName ? $paymentMethodName->getValue() : null;
    }

    /**
     * @return string
     */
    protected function getStoreName()
    {
        return substr(\XLite\Core\Config::getInstance()->Company->company_name, 0, 16);
    }

    /**
     * Check - payment processor is applicable for specified order or not
     *
     * @param \XLite\Model\Order          $order  Order
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return boolean
     */
    public function isApplicable(\XLite\Model\Order $order, \XLite\Model\Payment\Method $method)
    {
        $disabled = \Amazon\PayWithAmazon\Main::hasPaymentDisabledProducts($order);

        return parent::isApplicable($order, $method) && !$disabled;
    }
}
