<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon;

use XLite\Core\Cache\ExecuteCached;

/**
 * PayWithAmazon module main class
 */
abstract class Main extends \XLite\Module\AModule
{
    public const SP_IDS = [
        'USD' => 'A1PQFSSKP8TT2U',
        'GBP' => 'AWDB5T3U5TUAJ',
        'EUR' => 'A94PNDCDBJL2X'
    ];

    public const PLATFORM_IDS = [
        'USD' => 'A1PQFSSKP8TT2U',
        'GBP' => 'A1PQFSSKP8TT2U',
        'EUR' => 'A1PQFSSKP8TT2U'
    ];

    /**
     * @return Object|\XLite\Model\Payment\Method
     */
    public static function getMethod()
    {
        return ExecuteCached::executeCachedRuntime(static function () {
            return \XLite\Core\Database::getRepo('XLite\Model\Payment\Method')
                ->findByServiceName(self::getServiceName());
        }, [__CLASS__, __FUNCTION__]);
    }

    /**
     * @return \Amazon\PayWithAmazon\Model\Payment\Processor\PayWithAmazon
     */
    public static function getProcessor()
    {
        return static::getMethod()->getProcessor();
    }

    /**
     * @return array
     */
    public static function getAmazonUrls(): array
    {
        return [
            'payments.amazon.com',
            'payments-eu.amazon.com',
            'sellercentral.amazon.com',
            'sellercentral-europe.amazon.com'
        ];
    }

    /**
     * @return string
     */
    public static function getServiceName()
    {
        return 'PayWithAmazon';
    }

    /**
     * @param string $language
     *
     * @return string
     */
    public static function getLocaleCode()
    {
        $method = self::getMethod();
        $region = $method->getSetting('region');
        //for us-uk regions allow only these locales
        switch ($region) {
            case 'USD':
                return 'en_US';
            case 'GBP':
                return 'en_GB';
            default:
                $language = \XLite\Core\Session::getInstance()->getLanguage()->getCode();
                $locales = [
                    'en_US', 'en_GB', 'de_DE', 'fr_FR', 'it_IT', 'es_ES'
                ];

                $locale = array_filter($locales, static function ($item) use ($language) {
                    return strpos($item, strtolower($language)) === 0;
                });

                return count($locale) === 1 ? reset($locale) : 'en_US';
        }
    }

    /**
     * @param \Amazon\PayWithAmazon\Core\TokenManager $tokenManager
     *
     * @return array
     */
    public static function getAuthData(\Amazon\PayWithAmazon\Core\TokenManager $tokenManager): array
    {
        $method = self::getMethod();
        $region = \Amazon\PayWithAmazon\View\FormField\Select\Region::getRegionByCurrency($method->getSetting('region'));

        return [
            'public_key_id' => $method->getSetting('public_key_id'),
            'private_key'   => $tokenManager->decode($method->getSetting('private_key')),
            'region'        => $region,
            'sandbox'       => $method->getSetting('mode') === 'test',
            'store_id'      => $method->getSetting('client_id'),
        ];
    }

    /**
     * @return boolean
     */
    public static function isConfigured()
    {
        $method    = Main::getMethod();
        $processor = Main::getProcessor();

        return $processor->isConfigured($method) && $method->isEnabled();
    }

    /**
     * @param \XLite\Model\Cart|\XLite\Model\Order $cart
     *
     * @return boolean
     */
    public static function hasPaymentDisabledProducts($cart)
    {
        $disabled = false;
        foreach ($cart->getItems() as $item) {
            if (!$item->getProduct()->getAmazonPaymentEnabled()) {
                $disabled = true;

                break;
            }
        }

        return $disabled;
    }
}
