<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon\Controller\Customer;

use Amazon\PayWithAmazon\Main;
use XCart\Extender\Mapping\Extender;
use XLite\InjectLoggerTrait;

/**
 * @Extender\Mixin
 */
class Checkout extends \XLite\Controller\Customer\Checkout
{
    use InjectLoggerTrait;

    public static function defineFreeFormIdActions()
    {
        return array_merge(
            parent::defineFreeFormIdActions(),
            ['checkoutReview']
        );
    }

    protected function doActionCheckoutReview()
    {
        $data = \XLite\Core\Request::getInstance()->getData();
        $session = $data['amazonCheckoutSessionId'];

        $tokenManager = new \Amazon\PayWithAmazon\Core\TokenManager();
        $authData = \Amazon\PayWithAmazon\Main::getAuthData($tokenManager);
        $responseData = $this->getCheckoutRequest($session, $authData);

        if (empty($responseData['error'])) {
            $processor = Main::getProcessor();

            $shippingAddress = $processor->getAddressDataFromOrderReferenceDetails($responseData['shippingAddress']);
            $billingAddress = $processor->getAddressDataFromOrderReferenceDetails($responseData['billingAddress']);
            if ($shippingAddress || $billingAddress) {
                $this->updateAddress([
                    'shipping' => $shippingAddress ?? null,
                    'billing' => !empty($billingAddress) ? $billingAddress : $shippingAddress
                ]);

                if (
                    !\XLite\Core\Auth::getInstance()->isLogged()
                    && isset($responseData['buyer'])
                ) {
                    $this->requestData += [
                        'email'          => $responseData['buyer']['email'],
                        'create_profile' => false
                    ];

                    $this->updateProfile();

                    $this->setCheckoutAvailable();
                }

                \XLite\Core\TopMessage::addInfo('Address is updated');

                $this->getCart()->setPaymentMethod(Main::getMethod());
                $transaction = $this->getCart()->getFirstOpenPaymentTransaction();

                $paymentData       = isset($responseData['paymentPreferences']) ? reset($responseData['paymentPreferences']) : null;
                $paymentDescriptor = $paymentData['paymentDescriptor'] ?? null;
                if ($paymentDescriptor) {
                    $transaction->setDataCell('paymentMethodName', $paymentDescriptor, 'AmazonPaymentName', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
                }

                $transaction->setDataCell('checkoutSessionId', $session, 'amazonCheckoutSessionId', \XLite\Model\Payment\TransactionData::ACCESS_CUSTOMER);
                $transaction->update();
            } else {
                $this->getLogger('Amazon-PayWithAmazon')->error('Address update error: ' . __FUNCTION__, $responseData);

                \XLite\Core\TopMessage::addInfo('Address update failed');
            }
        } else {
            \XLite\Core\TopMessage::addError('Get amazon session error');
        }

        $this->setReturnURL($this->buildURL('checkout', '', ['amazon_returned' => true]));
    }

    /**
     * @param array $address
     */
    public function updateAddress($address)
    {
        $this->requestData['shippingAddress'] = $address['shipping'];
        $this->requestData['billingAddress']  = $address['billing'];
        $this->requestData['same_address'] = $address['shipping'] == $address['billing'];

        $this->updateShippingAddress();
        $this->updateBillingAddress();
    }

    protected function getCheckoutRequest(string $id, array $authData)
    {
        try {
            $client = new \Amazon\Pay\API\Client($authData);
            $result = $client->getCheckoutSession($id);
            $sessionResult = json_decode($result['response'], true);

            if ($result['status'] === 200) {
                $response = $sessionResult;
            } else {
                $response = [
                    'error' => $sessionResult['message'] ?? __FUNCTION__,
                    'method' => __FUNCTION__,
                    'status' => $result['status'],
                    'response' => $sessionResult
                ];

                $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
            }
        } catch (\Exception $e) {
            $response = [
                'error' => $e->getMessage(),
                'method' => __FUNCTION__,
            ];

            $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
        }

        return $response;
    }

    /**
     * Check if customer is returned from checkoutReviewReturnUrl
     *
     * @return boolean
     */
    public function isReturnedAfterAmazonCheckout()
    {
        return \XLite\Core\Request::getInstance()->amazon_returned === '1';
    }

    /**
     * Returned checkout should be have checkoutSessionId
     *
     * @return boolean
     */
    public function getAmazonSessionId()
    {
        $transaction = $this->getCart()->getFirstOpenPaymentTransaction();
        $sessionId = $transaction ? $transaction->getDataCell('checkoutSessionId') : null;

        return $sessionId ? $sessionId->getValue() : null;
    }
}
