<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon\Controller\Customer;

use XCart\Extender\Mapping\Extender;
use Amazon\PayWithAmazon\Main;
use XLite\InjectLoggerTrait;

/**
 * @Extender\Mixin
 */
class Cart extends \XLite\Controller\Customer\Cart
{
    use InjectLoggerTrait;

    protected function doActionCreateSession()
    {
        $postData = \XLite\Core\Request::getInstance()->getData();
        $tokenManager = new \Amazon\PayWithAmazon\Core\TokenManager();
        $authData = \Amazon\PayWithAmazon\Main::getAuthData($tokenManager);

        if ($postData['type'] === 'Login') {
            $payload = [
                'storeId' => $authData['store_id'],
                'signInReturnUrl' => $this->buildFullURL(
                    'amazon_login',
                    null,
                    [
                        'mode' => 'loginWithAmazon',
                        'returnUrl' => $this->getReferrerURL()
                    ]
                ),
                'signInScopes' => [
                    'name',
                    'email',
                    'postalCode',
                    'shippingAddress',
                    'billingAddress',
                    'phoneNumber',
                ]
            ];

            $payload = json_encode($payload, JSON_UNESCAPED_SLASHES);
        } elseif ($postData['type'] === 'ProcessOrder') {
            $transaction = $this->getCart()->getFirstOpenPaymentTransaction();
            if ($transaction) {
                $processor = Main::getProcessor();

                $shippingModifier = $this->getCart()->getModifier(\XLite\Model\Base\Surcharge::TYPE_SHIPPING, 'SHIPPING');
                $payload = $processor->getPayloadData($transaction, $shippingModifier && $shippingModifier->canApply());
                $payload['storeId'] = $authData['store_id'];
                $payload['webCheckoutDetails']['checkoutMode'] = 'ProcessOrder';

                $payload = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_FORCE_OBJECT);
            } else {
                $this->getLogger('Amazon-PayWithAmazon')->error('Transaction not found in ProcessOrder', [
                    'cart' => $this->getCart()->getOrderId(),
                    'postData' => $postData
                ]);

                $payload = [];
            }
        } else {
            $payload = [
                'storeId' => $authData['store_id'],
                'webCheckoutDetails' => [
                    'checkoutReviewReturnUrl' => $this->buildFullURL(
                        'checkout',
                        'checkoutReview'
                    ),
                ],
                "scopes" => ["name", "email", "phoneNumber", "billingAddress"],
            ];

            $notAllowed = \XLite\Core\Database::getRepo('XLite\Model\Country')?->findDisabledCodes();
            if ($notAllowed) {
                $codeArray = array_column($notAllowed, 'code');
                $payload['deliverySpecifications'] = [
                    'addressRestrictions' => [
                        'type' => 'NotAllowed',
                        'restrictions' => array_fill_keys($codeArray, [])
                    ]
                ];
            }

            $payload = json_encode($payload, JSON_UNESCAPED_SLASHES);
        }

        $this->printAJAX($this->generateButtonRequest($payload, $authData));
    }

    protected function generateButtonRequest($payload, array $authData)
    {
        try {
            $client = new \Amazon\Pay\API\Client($authData);
            $signature = $client->generateButtonSignature($payload);
            $response = [
                'signature'   => $signature,
                'payloadJSON' => $payload,
                'publicKeyId' => $authData['public_key_id']
            ];
        } catch (\Exception $e) {
            $response = [
                'error'  => $e->getMessage(),
                'method' => __FUNCTION__,
            ];

            $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
        }

        return $response;
    }
}
