<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon\Controller\Customer;

use Amazon\PayWithAmazon\Main;
use XLite\Core\Auth;
use XLite\Core\Database;
use XLite\Core\Request;
use XLite\Core\TopMessage;
use XLite\InjectLoggerTrait;
use XLite\Model\Address;
use XLite\Model\Profile;

/**
 * Amazon checkout controller
 */
class AmazonLogin extends \XLite\Controller\Customer\Checkout
{
    use InjectLoggerTrait;

    /**
     * Can be removed without extends checkout, if prepareAddressData define as static
     *
     * @return void
     */
    public function handleRequest()
    {
        \XLite\Controller\Customer\ACustomer::handleRequest();
    }

    protected function doNoAction()
    {
        if (Request::getInstance()->mode === 'loginWithAmazon') {
            $this->doActionLogin();
        } else {
            $this->setReturnURL($this->buildFullURL('main'));
        }
    }

    protected function doActionLogin()
    {
        $requestProcessed = false;
        $returnURL        = '';
        $error            = '';

        $data = \XLite\Core\Request::getInstance()->getData();
        $tokenManager = new \Amazon\PayWithAmazon\Core\TokenManager();
        $authData = \Amazon\PayWithAmazon\Main::getAuthData($tokenManager);
        $profileInfo = $this->getBuyerRequest($data['buyerToken'], $authData);

        if ($profileInfo && !empty($profileInfo['buyerId']) && !empty($profileInfo['email'])) {
            if (Request::getInstance()->mode === 'loginWithAmazon') {
                $profile = $this->getSocialLoginProfile(
                    $profileInfo['email'],
                    'Amazon',
                    $profileInfo['buyerId']
                );

                if ($profile) {
                    if (
                        !Auth::getInstance()->isLogged()
                        || Auth::getInstance()->getProfile()?->getProfileId() !== $profile->getProfileId()
                    ) {
                        if ($profile->isEnabled()) {
                            Auth::getInstance()->loginProfile($profile);

                            // We merge the logged in cart into the session cart
                            $profileCart = $this->getCart();
                            $profileCart->login($profile);

                            $this->fillAddressByResponse($profile, $profileInfo);

                            Database::getEM()->flush();

                            if ($profileCart->isPersistent()) {
                                $this->updateCart();
                            }
                        } else {
                            TopMessage::addError('Profile is disabled');
                            $returnURL = $this->getAuthReturnURL(true);
                        }
                    }
                } else {
                    \XLite\Core\TopMessage::addWarning(
                        'The email you tried to use is already registered in our store. Please sign in the classic way.'
                    );
                    $returnURL = $this->getAuthReturnURL(true);
                }
            } else {
                if (!Auth::getInstance()->getProfile()) {
                    $this->getCartProfile()?->setLogin($profileInfo['email']);
                    Database::getEM()->flush();
                }
            }

            if (!$returnURL) {
                $returnURL = $this->getAuthReturnURL();
            }

            $requestProcessed = true;
        }

        if (!$requestProcessed) {
            $this->getLogger('Amazon-PayWithAmazon')->error('Error: ' . __FUNCTION__, [
                'error'   => $error ?: 'We were unable to process this request',
            ]);
            TopMessage::addError('We were unable to process this request');
            $returnURL = $this->getAuthReturnURL(true);
        }

        $this->setReturnURL($returnURL);
    }

    protected function getBuyerRequest($buyerToken, array $authData)
    {
        try {
            $client = new \Amazon\Pay\API\Client($authData);
            $result = $client->getBuyer($buyerToken);
            $buyerResult = json_decode($result['response'], true);

            if ($result['status'] === 200) {
                $response = $buyerResult;
            } else {
                $response = [
                    'error' => $buyerResult['message'] ?? __FUNCTION__,
                    'method' => __FUNCTION__,
                    'status' => $result['status'],
                    'response' => $buyerResult
                ];

                $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
            }
        } catch (\Exception $e) {
            $response = [
                'error' => $e->getMessage(),
                'method' => __FUNCTION__,
            ];

            $this->getLogger('Amazon-PayWithAmazon')->error($response['error'], $response);
        }

        return $response;
    }

    /**
     * Fetches an existing social login profile or creates new
     *
     * @param string $login          E-mail address
     * @param string $socialProvider SocialLogin auth provider
     * @param string $socialId       SocialLogin provider-unique id
     *
     * @return Profile
     */
    protected function getSocialLoginProfile($login, $socialProvider, $socialId)
    {
        $profileRepo = Database::getRepo(Profile::class);
        $profile = $profileRepo?->findOneBy(
            [
                'socialLoginProvider' => $socialProvider,
                'socialLoginId'       => $socialId,
                'order'               => null,
            ]
        );

        if (!$profile) {
            $profile = $profileRepo?->findOneBy(['login' => $login, 'order' => null, 'anonymous' => false]);
        }

        if (!$profile) {
            $profile = new Profile();
            $profile->setLogin($login);
            $profile->create();
        } elseif ($profile->isAdmin()) {
            $profile = null;
        }

        if ($profile) {
            $profile->setSocialLoginProvider($socialProvider);
            $profile->setSocialLoginId($socialId);
        }

        return $profile;
    }

    /**
     * Set redirect URL
     *
     * @param mixed $failure Indicates if auth process failed OPTIONAL
     *
     * @return string
     */
    protected function getAuthReturnURL($failure = false)
    {
        if ($failure) {
            return $this->buildURL('login');
        }

        return Request::getInstance()->returnUrl ?: $this->buildURL('checkout');
    }

    /**
     * @param Profile $profile
     * @param array $responseData
     */
    public function fillAddressByResponse($profile, $responseData)
    {
        $this->setProfileAddress($profile, $responseData['shippingAddress']);
        $this->setProfileAddress($profile, $responseData['billingAddress'], true);
    }

    /**
     * @param Profile $profile
     * @param array $responseAddressData
     * @param bool $isBillingType
     */
    public function setProfileAddress($profile, $responseAddressData, $isBillingType = false)
    {
        $processor = Main::getProcessor();
        $addressData = $processor->getAddressDataFromOrderReferenceDetails($responseAddressData);

        if ($addressData) {
            if ($isBillingType) {
                \XLite\Core\Session::getInstance()->same_address = false;
            }

            $address = new \XLite\Model\Address();
            $address->map($this->prepareAddressData($addressData, $isBillingType ? 'billing' : 'shipping'));

            foreach ($profile->getAddresses() as $existingAddress) {
                /** @var Address $existingAddress */
                if ($existingAddress->isEqualAddress($address)) {
                    $address = $existingAddress;
                    break;
                }
            }

            $address->setProfile($profile);
            if ($isBillingType) {
                $profile->setBillingAddress($address);
            } else {
                $profile->setShippingAddress($address);
            }

            \XLite\Core\Database::getEM()->persist($address);
        }
    }
}
