<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Amazon\PayWithAmazon\Controller\Admin;

use XCart\Extender\Mapping\Extender;
use XLite\Core\Converter;

/**
 * @Extender\Mixin
 */
class PaymentMethod extends \XLite\Controller\Admin\PaymentMethod
{
    /**
     * Update payment method
     *
     * @return void
     */
    protected function doActionUpdate()
    {
        $method = $this->getPaymentMethod();

        if ($method && $method->getProcessor() instanceof \Amazon\PayWithAmazon\Model\Payment\Processor\PayWithAmazon) {
            $requestData  = \XLite\Core\Request::getInstance()->getData();
            $oldToken     = $method->getSetting('private_key_hash');
            $newToken     = $requestData['settings']['private_key_hash'] ?? null;
            $isKeyChanged = $newToken !== null && $newToken !== $oldToken;

            if ($isKeyChanged && !$this->isPrivateToken($newToken)) {
                \XLite\Core\TopMessage::addError('Private key you have provided is incorrect');

                $this->setReturnURL(
                    Converter::buildURL(
                        'payment_method',
                        null,
                        ['method_id' => $method->getMethodId()]
                    )
                );

                $isKeyChanged = false;
            } else {
                parent::doActionUpdate();
            }

            if ($isKeyChanged) {
                // actually encode after parent update only
                $tokenManager = new \Amazon\PayWithAmazon\Core\TokenManager();
                $encoded      = $tokenManager->encode($newToken);
                $method->setSetting('private_key', $encoded);
                $method->setSetting('private_key_hash', md5($encoded));

                $method->update();
            }

            if (!$method->isConfigured() && !\XLite\Core\Config::getInstance()->Security->customer_security) {
                \XLite\Core\TopMessage::addWarning(
                    'The "Pay with Amazon" feature requires https to be properly set up for your store.',
                    [
                        'url' => \XLite\Core\Converter::buildURL('https_settings'),
                    ]
                );
            }
        } else {
            parent::doActionUpdate();
        }
    }

    /**
     * @return boolean
     */
    public function isPrivateToken(string $token)
    {
        $starts = str_starts_with($token, '-----BEGIN PRIVATE KEY-----');
        $ends = str_ends_with($token, '-----END PRIVATE KEY-----');

        return $starts && $ends;
    }

    /**
     * @return boolean
     */
    public function isHttpsEnabled()
    {
        return \XLite\Core\Config::getInstance()->Security->customer_security;
    }
}
