<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\View;

use Includes\Utils\Module\Manager;
use Includes\Utils\Module\Module;
use XCart\Domain\ModuleManagerDomain;

/**
 * Module banner
 */
class ModuleBanner extends \XLite\View\AView
{
    public const PARAM_MODULE_NAME = 'moduleName';
    public const PARAM_CAN_CLOSE   = 'canClose';

    protected ?ModuleManagerDomain $moduleManagerDomain;

    public function __construct(array $params = [])
    {
        $this->moduleManagerDomain = \XCart\Container::getContainer()?->get(ModuleManagerDomain::class);

        parent::__construct($params);
    }

    /**
     * Returns CSS files
     *
     * @return array
     */
    public function getCSSFiles()
    {
        $list   = parent::getCSSFiles();
        $list[] = 'module_banner/style.css';

        return $list;
    }

    /**
     * Returns JS files
     *
     * @return array
     */
    public function getJSFiles()
    {
        $list   = parent::getJSFiles();
        $list[] = 'module_banner/controller.js';

        return $list;
    }

    /**
     * Return widget default template
     *
     * @return string
     */
    protected function getDefaultTemplate()
    {
        return 'module_banner/body.twig';
    }

    /**
     * Define widget parameters
     *
     * @return void
     */
    protected function defineWidgetParams()
    {
        parent::defineWidgetParams();

        $this->widgetParams += [
            static::PARAM_MODULE_NAME => new \XLite\Model\WidgetParam\TypeString('Module name', null),
            static::PARAM_CAN_CLOSE   => new \XLite\Model\WidgetParam\TypeBool('Can close', true),
        ];
    }

    /**
     * Check if widget is visible
     *
     * @return boolean
     */
    protected function isVisible()
    {
        return parent::isVisible()
            && !$this->isModuleInstalled()
            && !$this->isBannerClosed();
    }

    /**
     * Check module installed
     *
     * @return boolean
     */
    protected function isModuleInstalled()
    {
        [$author, $name] = Module::explodeModuleId($this->getModuleName());

        return $this->moduleManagerDomain->isEnabled("{$author}-{$name}");
    }

    /**
     * Get module name
     *
     * @return string
     */
    protected function getModuleName()
    {
        return $this->getParam(static::PARAM_MODULE_NAME);
    }

    /**
     * Get alphanumeric module name
     *
     * @return string
     */
    protected function getStringModuleName()
    {
        return str_replace('\\', '_', $this->getModuleName());
    }

    /**
     * Check can close
     *
     * @return boolean
     */
    protected function isCanClose()
    {
        return (bool) $this->getParam(static::PARAM_CAN_CLOSE);
    }

    /**
     * Check banner is closed
     *
     * @return boolean
     */
    protected function isBannerClosed()
    {
        $closedModuleBanners = \XLite\Core\TmpVars::getInstance()->closedModuleBanners ?: [];

        return $this->isCanClose() && !empty($closedModuleBanners[$this->getModuleName()]);
    }

    /**
     * Get style class
     *
     * @return string
     */
    protected function getStyleClass()
    {
        return strtolower($this->getStringModuleName());
    }

    /**
     * Returns ACR URL
     *
     * @return string
     */
    protected function getModuleURL()
    {
        [$author, $name] = explode('\\', $this->getModuleName());

        return Manager::getRegistry()->getModuleServiceURL($author, $name);
    }

    /**
     * Template method
     *
     * @return string
     */
    protected function getModuleBannerImageUrl()
    {
        return \XLite\Core\Layout::getInstance()->getResourceWebPath('module_banner/images/QSL_AbandonedCartReminder.png');
    }
}
