<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\View\ItemsList\Model;

use XLite\Core\Skin;
use XLite\Logic\ImageResize\Generator;
use XLite\Model\ImageSettings;

/**
 * Images settings items list widget
 */
class ImagesSettings extends \XLite\View\ItemsList\Model\Table
{
    public function getJSFiles()
    {
        $list = parent::getJSFiles();
        $list[] = 'images_settings/script.js';

        return $list;
    }

    /**
     * Define columns structure
     *
     * @return array
     */
    protected function defineColumns()
    {
        return [
            'name' => [
                static::COLUMN_NAME => static::t('Name'),
                static::COLUMN_TEMPLATE  => 'images_settings/table/cell.name.twig',
            ],
            'width' => [
                static::COLUMN_NAME      => static::t('Width (px)'),
                static::COLUMN_CLASS     => 'XLite\View\FormField\Inline\Input\Text\Integer',
                static::COLUMN_EDIT_ONLY => true,
                static::COLUMN_PARAMS    => [
                    \XLite\View\FormField\Input\Text\Base\Numeric::PARAM_MIN              => 0,
                    \XLite\View\FormField\Input\Text\Base\Numeric::PARAM_MOUSE_WHEEL_CTRL => false
                ],
            ],
            'height' => [
                static::COLUMN_NAME      => static::t('Height (px)'),
                static::COLUMN_CLASS     => 'XLite\View\FormField\Inline\Input\Text\Integer',
                static::COLUMN_EDIT_ONLY => true,
                static::COLUMN_PARAMS    => [
                    \XLite\View\FormField\Input\Text\Base\Numeric::PARAM_MIN              => 0,
                    \XLite\View\FormField\Input\Text\Base\Numeric::PARAM_MOUSE_WHEEL_CTRL => false
                ],
            ],
        ];
    }

    /**
     * Define repository name
     *
     * @return string
     */
    protected function defineRepositoryName()
    {
        return 'XLite\Model\ImageSettings';
    }

    /**
     * Return options list
     *
     * @param \XLite\Core\CommonCell $cnd       Search condition
     * @param boolean                $countOnly Return items list or only its size OPTIONAL
     *
     * @return array|integer
     */
    protected function getData(\XLite\Core\CommonCell $cnd, $countOnly = false)
    {
        $options = $this->prepareOptions(Skin::getInstance()->getCurrentImagesSettings());

        return $countOnly ? count($options) : $options;
    }

    /**
     * @param \XLite\Model\ImageSettings $entity
     *
     * @return bool
     */
    protected function isNameTooltipVisible($entity)
    {
        return in_array($entity->getCode(), [
            'LGThumbnailGrid',
            'LGThumbnailList'
        ], true);
    }

    /**
     * @param \XLite\Model\ImageSettings $entity
     *
     * @return string
     */
    protected function getNameTooltip($entity)
    {
        switch ($entity->getCode()) {
            case 'LGThumbnailGrid':
                $path = 'images/preview_grid.png';
                break;
            case 'LGThumbnailList':
                $path = 'images/preview_list.png';
                break;
            default:
                $path = '';
        }

        $url = \XLite\Core\Layout::getInstance()->getResourceWebPath($path);
        return "<img src='$url'>";
    }

    protected function sortOptions(array $options): array
    {
        $invoiceLogoOptions = null;

        /** @var ImageSettings $option */
        foreach ($options as $i => $option) {
            if (
                $option->getModel() === Generator::MODEL_LOGO
                && $option->getCode() === 'Invoice'
            ) {
                $invoiceLogoOptions = $option;
                unset($options[$i]);

                break;
            }
        }

        if ($invoiceLogoOptions) {
            $sortedOptions = [];

            foreach ($options as $option) {
                $sortedOptions[] = $option;

                // Invoice logo options should always go right after mobile logo options.
                if (
                    $option->getModel() === Generator::MODEL_LOGO
                    && $option->getCode() === 'Mobile'
                ) {
                    $sortedOptions[] = $invoiceLogoOptions;
                }
            }

            return $sortedOptions;
        }

        return $options;
    }

    /**
     * Correct options list
     *
     * @param array $options Options list (list of images sizes)
     *
     * @return array
     */
    protected function prepareOptions($options)
    {
        $options = $this->sortOptions($options);

        $editableSizes = Generator::getEditableImageSizes();
        $allSizes = Generator::getImageSizes();

        if ($editableSizes) {
            $needUpdate = false;

            // Prepare temporary array for checking
            $tmp = [];
            foreach ($editableSizes as $model => $opt) {
                foreach ($opt as $code) {
                    $tmp[sprintf('%s-%s', $model, $code)] = [
                        'model' => $model,
                        'code'  => $code,
                        'size'  => $allSizes[$model][$code],
                    ];
                }
            }

            $existingSizes = [];

            // Search for options which must be removed
            foreach ($options as $option) {
                $testKey = sprintf('%s-%s', $option->getModel(), $option->getCode());

                if (!isset($tmp[$testKey])) {
                    // Found size which is not present in the list of editable sizes - prepare to remove this
                    $toDelete[$option->getId()] = $option;
                    $needUpdate = true;
                } else {
                    // Keep option in the list of sizes presented in the database and not scheduled to be removed
                    $existingSizes[$testKey] = $option->getId();
                }
            }

            $moduleName = Skin::getInstance()->getCurrentSkinModuleId();

            // Search for image sizes which should be added to the database
            foreach ($tmp as $k => $v) {
                if (!isset($existingSizes[$k])) {
                    // Found an option which is not presented in the database - prepare to add this
                    $entity = new \XLite\Model\ImageSettings();
                    $entity->setModel($v['model']);
                    $entity->setCode($v['code']);
                    $entity->setModuleName($moduleName);
                    $entity->setWidth($v['size'][0]);
                    $entity->setHeight($v['size'][1]);

                    $toInsert[] = $entity;
                    $needUpdate = true;
                }
            }

            if (!empty($toInsert)) {
                // Create new image sizes
                \XLite\Core\Database::getRepo('XLite\Model\ImageSettings')->insertInBatch($toInsert);
            }

            if (!empty($toDelete)) {
                // Remove obsolete image sizes
                \XLite\Core\Database::getRepo('XLite\Model\ImageSettings')->deleteInBatchById($toDelete);
            }

            if ($needUpdate) {
                \XLite\Core\Database::getEM()->clear();
                $options = \XLite\Core\Database::getRepo('XLite\Model\ImageSettings')->findAll();
            }
        } else {
            $options = [];
        }

        return $options;
    }

    /**
     * Mark list as switchable (enable / disable)
     *
     * @return boolean
     */
    protected function isSwitchable()
    {
        return false;
    }

    /**
     * Mark list as non-removable
     *
     * @return boolean
     */
    protected function isRemoved()
    {
        return false;
    }

    /**
     * Mark list as sortable
     *
     * @return boolean
     */
    protected function getSortableType()
    {
        return static::SORT_TYPE_NONE;
    }

    /**
     * Get container class
     *
     * @return string
     */
    protected function getContainerClass()
    {
        return parent::getContainerClass() . ' images-settings';
    }

    /**
     * Get pager class name
     *
     * @return string
     */
    protected function getPagerClass()
    {
        return 'XLite\View\Pager\Admin\Model\Infinity';
    }

    /**
     * Disable specific sticky panel for item list
     *
     * @return boolean
     */
    protected function isPanelVisible()
    {
        return false;
    }
}
