<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\View\FormField\Input\Text;

use XLite\Core\PhoneNumber\PhoneNumberUtil;
use XLite\Core\PreloadedLabels\ProviderInterface;

class InternationalPhoneInput extends \XLite\View\FormField\Input\Text implements ProviderInterface
{
    public const INTL_FIELD_SUFFIX = '_intl_phone_number';
    public const NAME_HIDDEN_FIELD = 'hiddenName';
    public const NAME_ERROR_LBL_LIST = 'errorsList';
    public const NAME_ERROR_CODES = 'errorCodes';
    public const NAME_UTILS_SCRIPT = 'utilsScript';
    public const NAME_COUNTRY_CODE = 'countryCode';
    public const ADDITIONAL_WRAPPER_CLASS = 'wrapper-intl-phone-number';

    protected $vueAttributes = [
        'data-shortname'
    ];

    public function getJSFiles(): array
    {
        $files = parent::getJSFiles();
        if ($this->isShowEditAsIntlPhone()) {
            $files = array_merge(
                $files,
                [$this->getDir() . '/input/text/intl_phone.js']
            );
        }

        return $files;
    }

    protected function getFieldTemplate(): string
    {
        return $this->isShowEditAsIntlPhone()
            ? 'intl_phone.twig'
            : parent::getFieldTemplate();
    }

    protected function assembleClasses(array $classes): array
    {
        $result = parent::assembleClasses($classes);
        if ($this->isShowEditAsIntlPhone()) {
            $result = array_merge(
                $result,
                [
                    'not-significant',
                    'xc-intl-phone-number',
                    'field-intlPhoneNumber'
                ]
            );
        }

        return $result;
    }

    protected function getCommentedData(): array
    {
        $data = parent::getCommentedData();
        if ($this->isShowEditAsIntlPhone()) {
            $errorList = self::getWidgetErrorsList();

            $countryCode = 'us';
            if ($phoneNumber = $this->getValue()) {
                try {
                    $countryCode = PhoneNumberUtil::getInstance()->getISOCountryCode($phoneNumber);
                } catch (\Exception $e) {
                }
            }

            $data = $data + [
                self::NAME_HIDDEN_FIELD => $this->getName(),
                self::NAME_ERROR_LBL_LIST => self::getWidgetErrorsList(),
                self::NAME_ERROR_CODES => array_keys($errorList),
                self::NAME_UTILS_SCRIPT => $this->getUtilsScriptPath(),
                self::NAME_COUNTRY_CODE => $countryCode
            ];
        }

        return $data;
    }

    protected function getCommonAttributes(): array
    {
        $attributes = parent::getCommonAttributes();

        if ($this->isShowEditAsIntlPhone()) {
            $attributes['id'] .= self::INTL_FIELD_SUFFIX;
            $attributes['name'] = str_replace(['[', ']'], '', $attributes['name'])
                . self::INTL_FIELD_SUFFIX;
        }

        return $attributes;
    }

    public function getUtilsScriptPath(): string
    {
        return '/assets/web/common/intl_tel_input/js/utils.js';
    }

    public function getCommonFiles(): array
    {
        $files = parent::getCommonFiles();
        if ($this->isShowEditAsIntlPhone()) {
            $files = array_merge_recursive($files, [
                static::RESOURCE_CSS => [
                    'intl_tel_input/css/intlTelInput.min.css',
                    $this->getDir() . '/css/intl_phone.less'
                ],
                static::RESOURCE_JS => [
                    'intl_tel_input/js/utils.js',
                    'intl_tel_input/js/intlTelInput.min.js'
                ],
            ]);
        }

        return $files;
    }

    public static function getWidgetErrorsList(): array
    {
        return [
            -99 => 'Invalid phone number',
            0   => 'Invalid phone number',
            2   => 'Phone number is too short',
            3   => 'Phone number is too long',
            4   => 'Invalid phone number'
        ];
    }

    public static function getAdditionalWrapperClass(): string
    {
        return self::ADDITIONAL_WRAPPER_CLASS;
    }

    public function getWrapperClass(): string
    {
        return trim(parent::getWrapperClass() . ' ' . self::getAdditionalWrapperClass());
    }

    protected function getAttributes(): array
    {
        $attributes = parent::getAttributes();
        if ($this->isShowEditAsIntlPhone()) {
            foreach ($attributes as $name => $value) {
                if (in_array($name, $this->vueAttributes)) {
                    unset($attributes[$name]);
                }
            }

            $attributes['data-shortname'] = 'phone_intl_phone_number';
        }

        return $attributes;
    }

    protected function getVueAttributes(): array
    {
        $result = [];
        $attributes = parent::getAttributes();
        foreach ($this->vueAttributes as $name) {
            if (isset($attributes[$name])) {
                $result[$name] = $attributes[$name];
            }
        }

        return $result;
    }

    protected function getVueAttributesCode(): string
    {
        return ' ' . static::convertToHtmlAttributeString($this->getVueAttributes());
    }

    public function getPreloadedLanguageLabels()
    {
        $list = [];

        foreach (static::getWidgetErrorsList() as $value) {
            $list[$value] = static::t($value);
        }

        return $list;
    }

    public function isShowEditAsIntlPhone()
    {
        return !\XLite::isAdminZone();
    }

    protected function getFieldLabelTemplate()
    {
        return $this->isShowEditAsIntlPhone()
            ? 'form_field/form_field_label_intl_phone.twig'
            : parent::getFieldLabelTemplate();
    }
}
