<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\View;

use Includes\Utils\ArrayManager;
use XLite\Core\PreloadedLabels\ProviderInterface;

/**
 * \XLite\View\Address
 */
class Address extends \XLite\View\Dialog implements ProviderInterface
{
    /**
     * Widget parameter names
     */
    public const PARAM_DISPLAY_MODE    = 'displayMode';
    public const PARAM_ADDRESS         = 'address';
    public const PARAM_DISPLAY_WRAPPER = 'displayWrapper';

    /**
     * Allowed display modes
     */
    public const DISPLAY_MODE_TEXT = 'text';
    public const DISPLAY_MODE_FORM = 'form';

    /**
     * Service constants for schema definition
     */
    public const SCHEMA_CLASS    = 'class';
    public const SCHEMA_LABEL    = 'label';
    public const SCHEMA_REQUIRED = 'required';

    /**
     * Schema
     *
     * @var array
     */
    protected $schema = [];

    // {{{ Schema fields

    /**
     * Returns address param
     *
     * @return \XLite\Model\Address
     */
    protected function getAddress()
    {
        return $this->getParam(static::PARAM_ADDRESS);
    }

    /**
     * Get schema fields
     *
     * @return array
     */
    public function getSchemaFields()
    {
        $result = $this->schema;

        $fields = \XLite\Core\Database::getRepo('XLite\Model\AddressField')
            ->search(new \XLite\Core\CommonCell(['enabled' => true]));

        foreach ($fields as $field) {
            $result[$field->getServiceName()] = [
                static::SCHEMA_CLASS    => $field->getSchemaClass(),
                static::SCHEMA_LABEL    => $field->getName(),
                static::SCHEMA_REQUIRED => $field->getRequired(),
            ];
        }

        return $this->getFilteredSchemaFields($result);
    }

    /**
     * Filter schema fields
     *
     * @param array $fields Schema fields to filter
     *
     * @return array
     */
    protected function getFilteredSchemaFields($fields)
    {
        if (!isset($fields['country_code'])) {
            // Country code field is disabled
            // We need leave oonly one state field: selector or text field

            $deleteStateSelector = true;

            $address = $this->getParam(self::PARAM_ADDRESS);

            if ($address && $address->getCountry() && $address->getCountry()->hasStates()) {
                $deleteStateSelector = false;
            }

            if ($deleteStateSelector) {
                unset($fields['state_id']);
            } else {
                unset($fields['custom_state']);
            }
        }

        $firstEnabledAddressLineField = '';
        foreach (['address1', 'address2', 'address3'] as $line) {
            if (isset($fields[$line])) {
                $firstEnabledAddressLineField = $line;
                break;
            }
        }

        $fields = ArrayManager::insertAfterKey($fields, $firstEnabledAddressLineField, 'address_line_concat_comma_divided', [
            'class'    => '\XLite\View\FormField\Input\Text',
            'label'    => static::t('Address'),
            'required' => true,
        ]);

        unset($fields['address1'], $fields['address2'], $fields['address3']);

        return $fields;
    }

    // }}}

    /**
     * Get field style
     *
     * @param string $fieldName Field name
     *
     * @return string
     */
    protected function getFieldStyle($fieldName)
    {
        $result = 'address-text-cell address-text-' . $fieldName;

        if (
            \XLite\Core\Database::getRepo('XLite\Model\AddressField')
                ->findOneBy(['serviceName' => $fieldName, 'additional' => true])
        ) {
            $result .= ' additional-field';
        }

        return $result;
    }

    /**
     * Get field value
     *
     * @param string  $fieldName    Field name
     * @param boolean $processValue Process value flag OPTIONAL
     *
     * @return string
     */
    public function getFieldValue($fieldName, $processValue = false)
    {
        $address = $this->getAddress();

        $methodName = 'get' . \Includes\Utils\Converter::convertToUpperCamelCase($fieldName);

        // $methodName assembled from 'get' + camelized $fieldName
        $result = $address->$methodName();

        if ($result && $processValue !== false) {
            switch ($fieldName) {
                case 'state_id':
                    $result = $address->getCountry() && $address->getCountry()->hasStates()
                        ? $address->getState()->getState()
                        : null;
                    break;

                case 'custom_state':
                    $result = $address->getCountry() && $address->getCountry()->hasStates()
                        ? null
                        : $result;
                    break;

                case 'country_code':
                    $result = $address->getCountry()
                        ? $address->getCountry()->getCountry()
                        : null;
                    break;

                case 'type':
                    $result = $address->getTypeName();
                    break;

                default:
            }
        }

        return $result;
    }

    /**
     * Get profile Id
     *
     * @return string
     */
    public function getProfileId()
    {
        return \XLite\Core\Request::getInstance()->profile_id;
    }

    /**
     * Get a list of JS files required to display the widget properly
     *
     * @return array
     */
    public function getJSFiles()
    {
        $list = parent::getJSFiles();

        if ($this->getParam(self::PARAM_DISPLAY_WRAPPER)) {
            $list[] = 'form_field/select_country.js';
        }

        return $list;
    }

    /**
     * Register CSS files
     *
     * @return array
     */
    public function getCSSFiles()
    {
        $list = parent::getCSSFiles();
        $list[] = 'address/style.less';

        return $list;
    }

    /**
     * Return title
     *
     * @return string
     */
    protected function getHead()
    {
        return 'Address';
    }

    /**
     * Return templates directory name
     *
     * @return string
     */
    protected function getDir()
    {
        return 'address/' . $this->getParam(self::PARAM_DISPLAY_MODE);
    }

    /**
     * Define widget parameters
     *
     * @return void
     */
    protected function defineWidgetParams()
    {
        parent::defineWidgetParams();

        $this->widgetParams += [
            self::PARAM_DISPLAY_MODE => new \XLite\Model\WidgetParam\TypeString(
                'Display mode',
                self::DISPLAY_MODE_TEXT,
                false
            ),
            self::PARAM_ADDRESS => new \XLite\Model\WidgetParam\TypeObject(
                'Address object',
                new \XLite\Model\Address(),
                false
            ),
            self::PARAM_DISPLAY_WRAPPER => new \XLite\Model\WidgetParam\TypeBool(
                'Display wrapper',
                false,
                false
            ),
        ];
    }

    /**
     * Get default template
     *
     * @return string
     */
    protected function getDefaultTemplate()
    {
        return 'address/wrapper.twig';
    }

    /**
     * Return internal list name
     *
     * @return string
     */
    protected function getListName()
    {
        return 'address.' . $this->getParam(self::PARAM_DISPLAY_MODE);
    }

    /**
     * Use body template
     *
     * @return boolean
     */
    protected function useBodyTemplate()
    {
        return !$this->getParam(self::PARAM_DISPLAY_WRAPPER);
    }

    /**
     * Array of labels in following format.
     *
     * 'label' => 'translation'
     *
     * @return mixed
     */
    public function getPreloadedLanguageLabels()
    {
        return [
            'Select one' => static::t('Select one')
        ];
    }
}
