<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model\Shipping;

use Doctrine\ORM\Mapping as ORM;

/**
 * Shipping markup model
 *
 * @ORM\Entity
 * @ORM\Table (name="shipping_markups",
 *      indexes={
 *          @ORM\Index (name="rate", columns={"method_id","zone_id","min_weight","min_total","min_discounted_total","min_items"}),
 *          @ORM\Index (name="max_weight", columns={"max_weight"}),
 *          @ORM\Index (name="max_total", columns={"max_total"}),
 *          @ORM\Index (name="max_discounted_total", columns={"max_discounted_total"}),
 *          @ORM\Index (name="max_items", columns={"max_items"}),
 *          @ORM\Index (name="markup_flat", columns={"markup_flat"}),
 *          @ORM\Index (name="markup_per_item", columns={"markup_per_item"}),
 *          @ORM\Index (name="markup_percent", columns={"markup_percent"}),
 *          @ORM\Index (name="markup_per_weight", columns={"markup_per_weight"})
 *      }
 * )
 */
class Markup extends \XLite\Model\AEntity
{
    public const INFINITY_VALUE = 999999999;

    /**
     * A unique ID of the markup
     *
     * @var integer
     *
     * @ORM\Id
     * @ORM\GeneratedValue (strategy="AUTO")
     * @ORM\Column (type="integer")
     */
    protected $markup_id;

    /**
     * Markup condition: min weight of products in the order
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=4)
     */
    protected $min_weight = 0;

    /**
     * Markup condition: max weight of products in the order
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=4)
     */
    protected $max_weight = self::INFINITY_VALUE;

    /**
     * Markup condition: min order subtotal
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $min_total = 0;

    /**
     * Markup condition: max order subtotal
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $max_total = self::INFINITY_VALUE;

    /**
     * Markup condition: min discounted order subtotal
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $min_discounted_total = 0;

    /**
     * Markup condition: max discounted order subtotal
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $max_discounted_total = self::INFINITY_VALUE;

    /**
     * Markup condition: min product items in the order
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=0)
     */
    protected $min_items = 0;

    /**
     * Markup condition: max product items in the order
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=0)
     */
    protected $max_items = self::INFINITY_VALUE;

    /**
     * Markup value: flat rate value
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $markup_flat = 0;

    /**
     * Markup value: percent value
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $markup_percent = 0;

    /**
     * Markup value: flat rate value per product item
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $markup_per_item = 0;

    /**
     * Markup value: flat rate value per weight unit
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=2)
     */
    protected $markup_per_weight = 0;

    /**
     * Shipping method (relation)
     *
     * @var \XLite\Model\Shipping\Method
     *
     * @ORM\ManyToOne  (targetEntity="XLite\Model\Shipping\Method", inversedBy="shipping_markups")
     * @ORM\JoinColumn (name="method_id", referencedColumnName="method_id", onDelete="CASCADE")
     */
    protected $shipping_method;

    /**
     * Zone (relation)
     *
     * @var \XLite\Model\Zone
     *
     * @ORM\ManyToOne  (targetEntity="XLite\Model\Zone", inversedBy="shipping_markups", cascade={"merge"})
     * @ORM\JoinColumn (name="zone_id", referencedColumnName="zone_id", onDelete="CASCADE")
     */
    protected $zone;

    /**
     * Calculated markup value
     *
     * @var float
     */
    protected $markupValue = 0;

    /**
     * getMarkupValue
     *
     * @return float
     */
    public function getMarkupValue()
    {
        return $this->markupValue;
    }

    /**
     * setMarkupValue
     *
     * @param integer $value Markup value
     *
     * @return void
     */
    public function setMarkupValue($value)
    {
        $this->markupValue = $value;
    }

    /**
     * Has rates
     *
     * @return boolean
     */
    public function hasRates()
    {
        return true;
    }

    /**
     * Returns weight range
     *
     * @return array
     */
    public function getWeightRange()
    {
        return [
            $this->getMinWeight(),
            $this->getMaxWeight() == static::INFINITY_VALUE ? html_entity_decode('&#x221E;') : $this->getMaxWeight()
        ];
    }

    /**
     * Set weight range
     *
     * @param array $value value
     *
     * @return array
     */
    public function setWeightRange($value)
    {
        if (is_array($value)) {
            $this->setMinWeight($value[0]);
            $this->setMaxWeight($value[1] === html_entity_decode('&#x221E;') ? static::INFINITY_VALUE : $value[1]);
        }
    }

    /**
     * Returns subtotal range
     *
     * @return array
     */
    public function getSubtotalRange()
    {
        return [
            $this->getMinTotal(),
            $this->getMaxTotal() == static::INFINITY_VALUE ? html_entity_decode('&#x221E;') : $this->getMaxTotal()
        ];
    }

    /**
     * Set subtotal range
     *
     * @param array $value value
     *
     * @return array
     */
    public function setSubtotalRange($value)
    {
        if (is_array($value)) {
            $this->setMinTotal($value[0]);
            $this->setMaxTotal($value[1] === html_entity_decode('&#x221E;') ? static::INFINITY_VALUE : $value[1]);
        }
    }

    /**
     * Returns discounted subtotal range
     *
     * @return array
     */
    public function getDiscountedSubtotalRange()
    {
        return [
            $this->getMinDiscountedTotal(),
            $this->getMaxDiscountedTotal() == static::INFINITY_VALUE ? html_entity_decode('&#x221E;') : $this->getMaxDiscountedTotal()
        ];
    }

    /**
     * Set discounted subtotal range
     *
     * @param array $value value
     *
     * @return array
     */
    public function setDiscountedSubtotalRange($value)
    {
        if (is_array($value)) {
            $this->setMinDiscountedTotal($value[0]);
            $this->setMaxDiscountedTotal($value[1] === html_entity_decode('&#x221E;') ? static::INFINITY_VALUE : $value[1]);
        }
    }

    /**
     * Returns items range
     *
     * @return array
     */
    public function getItemsRange()
    {
        return [
            $this->getMinItems(),
            $this->getMaxItems() == static::INFINITY_VALUE ? html_entity_decode('&#x221E;') : $this->getMaxItems()
        ];
    }

    /**
     * Set items range
     *
     * @param array $value value
     *
     * @return array
     */
    public function setItemsRange($value)
    {
        if (is_array($value)) {
            $this->setMinItems($value[0]);
            $this->setMaxItems($value[1] === html_entity_decode('&#x221E;') ? static::INFINITY_VALUE : $value[1]);
        }
    }

    /**
     * Get markup_id
     *
     * @return integer
     */
    public function getMarkupId()
    {
        return $this->markup_id;
    }

    /**
     * Set min_weight
     *
     * @param float $minWeight
     * @return Markup
     */
    public function setMinWeight($minWeight)
    {
        $this->min_weight = $minWeight;
        return $this;
    }

    /**
     * Get min_weight
     *
     * @return float
     */
    public function getMinWeight()
    {
        return $this->min_weight;
    }

    /**
     * Set max_weight
     *
     * @param float $maxWeight
     * @return Markup
     */
    public function setMaxWeight($maxWeight)
    {
        $this->max_weight = $maxWeight;
        return $this;
    }

    /**
     * Get max_weight
     *
     * @return float
     */
    public function getMaxWeight()
    {
        return $this->max_weight;
    }

    /**
     * Set min_total
     *
     * @param float $minTotal
     * @return Markup
     */
    public function setMinTotal($minTotal)
    {
        $this->min_total = $minTotal;
        return $this;
    }

    /**
     * Get min_total
     *
     * @return float
     */
    public function getMinTotal()
    {
        return $this->min_total;
    }

    /**
     * Set max_total
     *
     * @param float $maxTotal
     * @return Markup
     */
    public function setMaxTotal($maxTotal)
    {
        $this->max_total = $maxTotal;
        return $this;
    }

    /**
     * Get max_total
     *
     * @return float
     */
    public function getMaxTotal()
    {
        return $this->max_total;
    }

    /**
     * Set min_discounted_total
     *
     * @param float $minDiscountedTotal
     * @return Markup
     */
    public function setMinDiscountedTotal($minDiscountedTotal)
    {
        $this->min_discounted_total = $minDiscountedTotal;
        return $this;
    }

    /**
     * Get min_discounted_total
     *
     * @return float
     */
    public function getMinDiscountedTotal()
    {
        return $this->min_discounted_total;
    }

    /**
     * Set max_discounted_total
     *
     * @param float $maxDiscountedTotal
     * @return Markup
     */
    public function setMaxDiscountedTotal($maxDiscountedTotal)
    {
        $this->max_discounted_total = $maxDiscountedTotal;
        return $this;
    }

    /**
     * Get max_discounted_total
     *
     * @return float
     */
    public function getMaxDiscountedTotal()
    {
        return $this->max_discounted_total;
    }

    /**
     * Set min_items
     *
     * @param float $minItems
     * @return Markup
     */
    public function setMinItems($minItems)
    {
        $this->min_items = $minItems;
        return $this;
    }

    /**
     * Get min_items
     *
     * @return float
     */
    public function getMinItems()
    {
        return $this->min_items;
    }

    /**
     * Set max_items
     *
     * @param float $maxItems
     * @return Markup
     */
    public function setMaxItems($maxItems)
    {
        $this->max_items = $maxItems;
        return $this;
    }

    /**
     * Get max_items
     *
     * @return float
     */
    public function getMaxItems()
    {
        return $this->max_items;
    }

    /**
     * Set markup_flat
     *
     * @param float $markupFlat
     * @return Markup
     */
    public function setMarkupFlat($markupFlat)
    {
        $this->markup_flat = $markupFlat;
        return $this;
    }

    /**
     * Get markup_flat
     *
     * @return float
     */
    public function getMarkupFlat()
    {
        return $this->markup_flat;
    }

    /**
     * Set markup_percent
     *
     * @param float $markupPercent
     * @return Markup
     */
    public function setMarkupPercent($markupPercent)
    {
        $this->markup_percent = $markupPercent;
        return $this;
    }

    /**
     * Get markup_percent
     *
     * @return float
     */
    public function getMarkupPercent()
    {
        return $this->markup_percent;
    }

    /**
     * Set markup_per_item
     *
     * @param float $markupPerItem
     * @return Markup
     */
    public function setMarkupPerItem($markupPerItem)
    {
        $this->markup_per_item = $markupPerItem;
        return $this;
    }

    /**
     * Get markup_per_item
     *
     * @return float
     */
    public function getMarkupPerItem()
    {
        return $this->markup_per_item;
    }

    /**
     * Set markup_per_weight
     *
     * @param float $markupPerWeight
     * @return Markup
     */
    public function setMarkupPerWeight($markupPerWeight)
    {
        $this->markup_per_weight = $markupPerWeight;
        return $this;
    }

    /**
     * Get markup_per_weight
     *
     * @return float
     */
    public function getMarkupPerWeight()
    {
        return $this->markup_per_weight;
    }

    /**
     * Set shipping_method
     *
     * @param \XLite\Model\Shipping\Method $shippingMethod
     * @return Markup
     */
    public function setShippingMethod(\XLite\Model\Shipping\Method $shippingMethod = null)
    {
        $this->shipping_method = $shippingMethod;
        return $this;
    }

    /**
     * Get shipping_method
     *
     * @return \XLite\Model\Shipping\Method
     */
    public function getShippingMethod()
    {
        return $this->shipping_method;
    }

    /**
     * Set zone
     *
     * @param \XLite\Model\Zone $zone
     * @return Markup
     */
    public function setZone(\XLite\Model\Zone $zone = null)
    {
        $this->zone = $zone;
        return $this;
    }

    /**
     * Get zone
     *
     * @return \XLite\Model\Zone
     */
    public function getZone()
    {
        return $this->zone;
    }
}
