<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model\Repo\Order\Status;

use XLite\Model\Order\Status\AStatusTranslation;
use XLite\Model\Repo\ModelRepoTranslationTrait;

/**
 * Abstract order status repository
 */
class AStatus extends \XLite\Model\Repo\Base\I18n
{
    use ModelRepoTranslationTrait;

    /**
     * Default 'order by' field name
     *
     * @var string
     */
    protected $defaultOrderBy = 'position';

    /**
     * Alternative record identifiers
     *
     * @var array
     */
    protected $alternativeIdentifier = [
        ['code'],
    ];

    /**
     * Find entity by name (any language)
     *
     * @param string  $name      Name
     * @param boolean $countOnly Count only OPTIONAL
     *
     * @return \XLite\Model\AttributeGroup|integer
     */
    public function findOneByName($name, $countOnly = false)
    {
        return $countOnly
            ? count($this->defineOneByNameQuery($name)->getResult())
            : $this->defineOneByNameQuery($name)->getSingleResult();
    }

    /**
     * Define query builder for findOneByName() method
     *
     * @param string $name Name
     *
     * @return \Doctrine\ORM\QueryBuilder
     */
    protected function defineOneByNameQuery($name)
    {
        $qb = $this->createQueryBuilder()
            ->andWhere('translations.name = :name')
            ->setParameter('name', $name);

        return $qb;
    }

    public function getTranslationDataFromYamlStructure(array $entity, string $lngCode): ?array
    {
        if (
            (!empty($entity['code']))
            && ($result = $this->getTranslationFromYamlStructure($entity, $lngCode))
        ) {
            return [
                $entity['code'],
                $result['name'] ?? '',
                $result['customerName'] ?? ''
            ];
        }

        return null;
    }

    public function getTranslationDataFromModel(AStatusTranslation $translation): ?array
    {
        /** @var \Xlite\Model\Order\Status\AStatus $status */
        $status = $translation->getOwner();

        return [
            $status->getCode(),
            $translation->getName(),
            $translation->getCustomerName()
        ];
    }

    public function generateYamlEntityForTranslationModule(array $data, string $lngCode): ?array
    {
        if (empty($data[0])) {
            return null;
        }

        return [
            'code'         => $data[0],
            'translations' => [
                array_filter([
                    'code'         => $lngCode,
                    'name'         => $data[1] ?? '',
                    'customerName' => $data[2] ?? '',
                ]),
            ]
        ];
    }
}
