<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model\Repo;

use XLite\Model\NotificationTranslation;

/**
 * Notifications repository
 */
class Notification extends \XLite\Model\Repo\Base\I18n
{
    use ModelRepoTranslationTrait;

    /**
     * @var array
     */
    protected $alternativeIdentifier = [
        ['templatesDirectory'],
    ];

    protected $defaultOrderBy = 'position';

    /**
     * @return array
     */
    public function getModular()
    {
        return $this
            ->createOriginalQueryBuilder()
            ->where('n.module is NOT NULL')
            ->getResult();
    }

    public function disableNotificationsByModule(string $module): void
    {
        $this
            ->getQueryBuilder()
            ->update($this->_entityName, 'n')
            ->set('n.enabledForAdmin', 0)
            ->set('n.enabledForCustomer', 0)
            ->set('n.available', 0)
            ->where('n.module = :module')
            ->setParameter('module', $module)
            ->execute();
    }

    public function makeAvailableByModule(string $module): void
    {
        $this
            ->getQueryBuilder()
            ->update($this->_entityName, 'n')
            ->set('n.available', 1)
            ->where('n.module = :module')
            ->setParameter('module', $module)
            ->execute();
    }

    /**
     * @param string $alias
     * @param string $indexBy
     * @param string $code
     *
     * @return AQueryBuilder
     */
    public function createQueryBuilder($alias = null, $indexBy = null, $code = null)
    {
        return parent::createQueryBuilder($alias, $indexBy, $code)
            ->where('n.available = 1');
    }

    /**
     * @param string $alias
     * @param string $indexBy
     * @param string $code
     *
     * @return AQueryBuilder
     */
    public function createOriginalQueryBuilder($alias = null, $indexBy = null, $code = null)
    {
        return parent::createQueryBuilder($alias, $indexBy, $code);
    }

    public function getTranslationDataFromYamlStructure(array $entity, string $lngCode): ?array
    {
        if (
            (!empty($entity['templatesDirectory']))
            && ($result = $this->getTranslationFromYamlStructure($entity, $lngCode))
        ) {
            return [
                $entity['templatesDirectory'],
                $result['name'] ?? '',
                $result['description'] ?? '',
                $result['customerSubject'] ?? '',
                $result['customerText'] ?? '',
                $result['adminSubject'] ?? '',
                $result['adminText'] ?? ''
            ];
        }

        return null;
    }

    public function getTranslationDataFromModel(NotificationTranslation $translation): ?array
    {
        /** @var \Xlite\Model\Notification $notification */
        $notification = $translation->getOwner();

        return [
            $notification->getTemplatesDirectory(),
            $translation->getName(),
            $translation->getDescription(),
            $translation->getCustomerSubject(),
            $translation->getCustomerText(),
            $translation->getAdminSubject(),
            $translation->getAdminText()
        ];
    }

    public function generateYamlEntityForTranslationModule(array $data, string $lngCode): ?array
    {
        if (empty($data[0])) {
            return null;
        }

        return [
            'templatesDirectory' => $data[0],
            'translations'       => [
                array_filter([
                    'code'            => $lngCode,
                    'name'            => $data[1] ?? '',
                    'description'     => $data[2] ?? '',
                    'customerSubject' => $data[3] ?? '',
                    'customerText'    => $data[4] ?? '',
                    'adminSubject'    => $data[5] ?? '',
                    'adminText'       => $data[6] ?? '',
                ])
            ]
        ];
    }
}
