<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model\Repo;

use XLite\Model\CurrencyTranslation;

/**
 * Currency repository
 */
class Currency extends \XLite\Model\Repo\Base\I18n
{
    use ModelRepoTranslationTrait;

    /**
     * Repository type
     *
     * @var string
     */
    protected $type = self::TYPE_SERVICE;

    /**
     * Alternative record identifiers
     *
     * @var array
     */
    protected $alternativeIdentifier = [
        ['code'],
    ];


    /**
     * Find all used into orders currency
     *
     * @return array
     */
    public function findAllSortedByName()
    {
        return $this->defineAllSortedByNameQuery()->getResult();
    }

    /**
     * Find all used into orders currency
     *
     * @return array
     */
    public function findUsed()
    {
        return $this->defineFindUsedQuery()->getResult();
    }


    /**
     * Define query for findAllSortedByName() method
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineAllSortedByNameQuery()
    {
        return $this->createQueryBuilder('c')
            ->addSelect('translations')
            ->orderBy('translations.name');
    }

    /**
     * Define query for findUsed() method
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineFindUsedQuery()
    {
        return $this->createQueryBuilder('c')
            ->innerJoin('c.orders', 'o', 'WITH', 'o.order_id IS NOT NULL');
    }

    // }}}

    public function getTranslationDataFromYamlStructure(array $entity, string $lngCode): ?array
    {
        if (
            !empty($entity['currency_id'])
            && ($result = $this->getTranslationFromYamlStructure($entity, $lngCode))
            && !empty($result['name'])
        ) {
            return [
                $entity['currency_id'],
                $result['name']
            ];
        }

        return null;
    }

    public function getTranslationDataFromModel(CurrencyTranslation $translation): ?array
    {
        /** @var \XLite\Model\Currency $currency */
        $currency = $translation->getOwner();

        return [
            $currency->getCurrencyId(),
            $translation->getName()
        ];
    }

    public function generateYamlEntityForTranslationModule(array $data, string $lngCode): ?array
    {
        if (empty($data[0]) || empty($data[1])) {
            return null;
        }

        return [
            'currency_id' => $data[0],
            'translations' => [
                [
                    'code' => $lngCode,
                    'name' => $data[1]
                ]
            ]
        ];
    }
}
