<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model;

use DateTime;
use Doctrine\ORM\Mapping as ORM;
use JMS\Serializer\Annotation as JMS;
use Gesdinet\JWTRefreshTokenBundle\Model\RefreshTokenInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity
 * @ORM\Table(name="refresh_tokens")
 */
class RefreshToken extends AEntity implements RefreshTokenInterface
{
    /**
     * @Assert\Positive()
     * @ORM\Id()
     * @ORM\GeneratedValue(strategy="AUTO")
     * @ORM\Column(type="integer", options={"unsigned"=true})
     */
    private ?int $id;

    /**
     * @JMS\Type("string")
     * @ORM\Column(type="string", options={}, nullable=true)
     */
    private ?string $refreshToken = null;

    /**
     * @JMS\Type("string")
     * @ORM\Column(type="string", options={})
     */
    private string $username;

    /**
     * @ORM\Column(type="datetime", options={})
     */
    private DateTime $valid;

    public function getId(): ?int
    {
        return $this->id;
    }

    public function setId(?int $id): static
    {
        $this->id = $id;

        return $this;
    }

    public function getRefreshToken(): ?string
    {
        return $this->refreshToken;
    }

    public function getUsername(): string
    {
        return $this->username;
    }

    public function getValid(): DateTime
    {
        return $this->valid;
    }

    public function setUsername($username): static
    {
        $this->username = $username;

        return $this;
    }

    public function setValid($valid): static
    {
        $this->valid = $valid;

        return $this;
    }

    /**
     * Creates a new model instance based on the provided details.
     */
    public static function createForUserWithTtl(string $refreshToken, UserInterface $user, int $ttl): RefreshTokenInterface
    {
        $valid = new DateTime();
        $valid->modify('+' . $ttl . ' seconds');

        return (new RefreshToken())
            ->setRefreshToken($refreshToken)
            ->setUsername(method_exists($user, 'getUserIdentifier') ? $user->getUserIdentifier() : $user->getUsername())
            ->setValid($valid);
    }

    /**
     * @return string Refresh Token
     */
    public function __toString()
    {
        return $this->getRefreshToken() ?: '';
    }

    public function setRefreshToken($refreshToken = null)
    {
        if ($refreshToken === null || $refreshToken === '') {
            trigger_deprecation('gesdinet/jwt-refresh-token-bundle', '1.0', 'Passing an empty token to %s() to automatically generate a token is deprecated.', __METHOD__);

            $refreshToken = bin2hex(random_bytes(64));
        }

        $this->refreshToken = $refreshToken;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function isValid()
    {
        return $this->getValid() !== null && $this->getValid() >= new DateTime();
    }
}
