<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model\Product;

use Includes\Utils\ArrayManager;
use Serializable;
use XLite\Model\Cart;
use XLite\Model\Product;

/**
 * ProductStockAvailabilityPolicy encapsulates product availability logic (determines whether the product can be added to cart and what is the available amount).
 *
 * ProductStockAvailabilityPolicy is a serializable (to facilitate caching) class that defines an overridable (see ProductVariants and Wholesale) strategy of product availability calculation. Conceptually, ProductStockAvailabilityPolicy is similar to DDD's concept of value objects, though it is not exactly that.
 *
 * TODO: Can it be integrated into a Product entity as a value object in terms of DDD? Doctrine Embeddable?
 *
 * ProductStockAvailabilityPolicy determines whether the product can be added to a cart or not and what is the available amount. In the most basic scenario, a user can add a product to a cart when the available amount is greater than zero. The available amount, in turn, is defined as a product stock amount minus the amount that the user has already added to cart.
 */
class ProductStockAvailabilityPolicy implements Serializable
{
    public const PRODUCT_ID                = 'product_id';
    public const PRODUCT_AMOUNT            = 'product_amount';
    public const PRODUCT_INVENTORY_ENABLED = 'product_inventory_enabled';
    public const PRODUCT_DEFAULT_AMOUNT    = 'product_default_amount';

    /**
     * We call and use this as a data transfer object, though technically it is an array.
     *
     * @var array
     */
    protected $dto;

    /**
     * Construct a new ProductStockAvailabilityPolicy using the given Product instance.
     *
     * @param Product $product
     */
    public function __construct(Product $product)
    {
        $this->dto = $this->createDTO($product);
    }

    /**
     * Get product amount available for adding to cart.
     *
     * @param Cart $cart
     *
     * @return int
     */
    public function getAvailableAmount(Cart $cart)
    {
        return $this->dto[static::PRODUCT_INVENTORY_ENABLED]
            ? max(0, $this->dto[static::PRODUCT_AMOUNT])
            : $this->dto[static::PRODUCT_DEFAULT_AMOUNT];
    }

    /**
     * Get product amount in cart.
     *
     * @param Cart $cart
     *
     * @return int
     */
    public function getInCartAmount(Cart $cart)
    {
        $cartItems  = $cart->getItemsByProductId($this->dto[static::PRODUCT_ID]);
        $cartAmount = ArrayManager::sumObjectsArrayFieldValues($cartItems, 'getAmount', true);

        return max(0, $cartAmount);
    }

    /**
     * Check if all product items in cart
     *
     * @param Cart $cart
     *
     * @return boolean
     */
    public function isAllStockInCart(Cart $cart)
    {
        return $this->getAvailableAmount($cart) <= $this->getInCartAmount($cart);
    }

    /**
     * Check if product is out of stock
     *
     * @param Cart $cart
     *
     * @return bool
     */
    public function isOutOfStock(Cart $cart)
    {
        return $this->dto[static::PRODUCT_INVENTORY_ENABLED] && $this->getAvailableAmount($cart) <= 0;
    }

    /**
     * Create a data transfer object out of the Product instance.
     * This object should contain all of the data necessary for getAvailableAmount() & canAddToCart() methods to compute their value.
     *
     * @param Product $product
     *
     * @return array
     */
    protected function createDTO(Product $product)
    {
        return [
            static::PRODUCT_ID                => $product->getProductId(),
            static::PRODUCT_AMOUNT            => $product->getPublicAmount(),
            static::PRODUCT_INVENTORY_ENABLED => $product->getInventoryEnabled(),
            static::PRODUCT_DEFAULT_AMOUNT    => $product->getDefaultAmount(),
        ];
    }

    /**
     * (PHP 5 &gt;= 5.1.0)<br/>
     * String representation of object
     * @link http://php.net/manual/en/serializable.serialize.php
     * @return string the string representation of the object or null
     */
    public function serialize(): string
    {
        return serialize($this->dto);
    }

    /**
     * @return array
     * @link https://www.php.net/manual/en/language.oop5.magic.php#object.serialize
     */
    public function __serialize(): array
    {
        return [
            'dto' => $this->dto,
        ];
    }

    /**
     * (PHP 5 &gt;= 5.1.0)<br/>
     * Constructs the object
     * @link http://php.net/manual/en/serializable.unserialize.php
     *
     * @param string $serialized <p>
     *                           The string representation of the object.
     *                           </p>
     *
     * @return void
     */
    public function unserialize(string $serialized): void
    {
        $this->dto = unserialize($serialized, ['allowed_classes' => false]);
    }

    /**
     * @param array $data
     *
     * @return void
     * @link https://www.php.net/manual/en/language.oop5.magic.php#object.unserialize
     */
    public function __unserialize(array $data): void
    {
        $this->dto = $data['dto'];
    }
}
