<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Model\Base;

use Doctrine\ORM\Mapping as ORM;
use XLite\Core\ImageOperator;
use XLite\Core\Request;
use XLite\Core\TopMessage;
use XLite\InjectLoggerTrait;

/**
 * Image abstract store
 *
 * @ORM\MappedSuperclass
 */
abstract class Image extends \XLite\Model\Base\Storage
{
    use InjectLoggerTrait;

    public const RETINA_RATIO = 2;

    public const WEBP_MIME_TYPE = 'image/webp';

    public const WEBP_EXTENSION = 'webp';

    public const SVG_EXTENSION = 'svg';

    /**
     * MIME type to extension translation table
     *
     * @var array
     */
    protected static $types = [
        'image/jpeg'          => 'jpeg',
        'image/jpg'           => 'jpg',
        'image/gif'           => 'gif',
        'image/xpm'           => 'xpm',
        'image/gd'            => 'gd',
        'image/gd2'           => 'gd2',
        'image/wbmp'          => 'bmp',
        'image/bmp'           => 'bmp',
        'image/x-ms-bmp'      => 'bmp',
        'image/x-windows-bmp' => 'bmp',
        'image/png'           => 'png',
        'image/svg+xml'       => self::SVG_EXTENSION,
        self::WEBP_MIME_TYPE  => self::WEBP_EXTENSION
    ];

    /**
     * Extended MIME types
     *
     * @var array
     */
    protected static $extendedTypes = [
        'application/ico'          => 'ico',
        'image/ico'                => 'ico',
        'image/icon'               => 'ico',
        'image/vnd.microsoft.icon' => 'ico',
        'image/x-ico'              => 'ico',
        'image/x-icon'             => 'ico',
        'text/ico'                 => 'ico',
    ];

    protected const DEFAULT_FONT_SIZE_PIXELS = 16;

    /** @var array<string, float|int> $svgMeasurements */
    protected static array $svgMeasurements = [
        'px' => 1,
        'em' => self::DEFAULT_FONT_SIZE_PIXELS,
        'ex' => self::DEFAULT_FONT_SIZE_PIXELS / 2,
        'pt' => self::DEFAULT_FONT_SIZE_PIXELS / 12,
        'pc' => self::DEFAULT_FONT_SIZE_PIXELS,
        'in' => self::DEFAULT_FONT_SIZE_PIXELS * 6,
        'cm' => self::DEFAULT_FONT_SIZE_PIXELS / (2.54 / 6),
        'mm' => self::DEFAULT_FONT_SIZE_PIXELS / (25.4 / 6),
    ];

    /**
     * Width
     *
     * @var integer
     *
     * @ORM\Column (type="integer")
     */
    protected $width = 0;

    /**
     * Height
     *
     * @var integer
     *
     * @ORM\Column (type="integer")
     */
    protected $height = 0;

    /**
     * Image hash
     *
     * @var string
     *
     * @ORM\Column (type="string", options={ "fixed": true }, length=32, nullable=true)
     */
    protected $hash;

    /**
     * Is image need process or not
     *
     * @var boolean
     *
     * @ORM\Column (type="boolean")
     */
    protected $needProcess = true;

    /**
     * @var bool
     */
    protected $includeFilenameInHash = true;

    /**
     * Check file is image or not
     *
     * @return boolean
     */
    public function isImage()
    {
        return true;
    }

    /**
     * Get image URL for customer front-end
     *
     * @return string
     */
    public function getFrontURL()
    {
        return (!$this->getRepository()->isCheckImage() || $this->checkImageHash()) ? parent::getFrontURL() : null;
    }

    /**
     * Check - image hash is equal data from DB or not
     *
     * @return boolean
     */
    public function checkImageHash()
    {
        $result = true;

        if ($this->getHash()) {
            [$path, $isTempFile] = $this->getLocalPath();

            $hash = \Includes\Utils\FileManager::getHash($path, false, $this->includeFilenameInHash);

            if ($isTempFile) {
                \Includes\Utils\FileManager::deleteFile($path);
            }

            $result = $this->getHash() === $hash;
        }

        return $result;
    }

    public function getActualFileName()
    {
        if (!$this->isURL()) {
            return $this->getPath();
        }

        return preg_replace('/\.[^\.]*?$/', '.' . $this->getExtension(), $this->getFileName());
    }

    /**
     * Check - image is exists in DB or not
     * TODO - remove - old method
     *
     * @return boolean
     */
    public function isExists()
    {
        return !is_null($this->getId());
    }

    /**
     * Clone
     *
     * @return \XLite\Model\AEntity
     */
    public function cloneEntity()
    {
        $newEntity = parent::cloneEntity();

        return $this->processClonnedEntity($newEntity);
    }

    /**
     * @param \XLite\Model\AEntity $newEntity
     *
     * @return \XLite\Model\AEntity
     */
    protected function processClonnedEntity($newEntity)
    {
        if ($this->getStorageType() !== static::STORAGE_URL) {
            $newEntity->setPath('');
            // Clone local image (will be created new file with unique name)
            $newEntity->loadFromLocalFile($this->getStoragePath(), null, false);
        }

        return $newEntity;
    }

    public function getExtensionByMIME()
    {
        if (
            $this->getMime() === static::WEBP_MIME_TYPE
            && !ImageOperator::isWebpSupported()
        ) {
            $imageOperatorEngineType = ImageOperator::getEngineType();

            if ($imageOperatorEngineType !== ImageOperator::ENGINE_SIMPLE) {
                TopMessage::addWarning(
                    'Your image processing library does not support WebP format.',
                    [
                        'name' => ImageOperator::getEngineName()
                    ]
                );
            }
        }

        return parent::getExtensionByMIME();
    }

    /**
     * Update file path - change file extension taken from MIME information.
     *
     * @return boolean
     */
    protected function updatePathByMIME()
    {
        $result = parent::updatePathByMIME();

        if ($result && !$this->isURL()) {
            [$path,] = $this->getLocalPath();

            $newExtension = $this->getExtensionByMIME();
            $pathinfo = pathinfo($path);

            if (!$newExtension) {
                $this->loadErrorMessage = [
                    'The file extension is forbidden ({{file}})',
                    ['file' => $this->getFileName()],
                ];

                return false;
            }

            $extension = $pathinfo['extension'] ?? '';

            // HARDCODE for BUG-2520
            if (strtolower($extension) === 'jpg' && $newExtension === 'jpeg') {
                $newExtension = 'jpg';
            }

            if ($newExtension !== $extension) {
                $newPath = \Includes\Utils\FileManager::getUniquePath(
                    $pathinfo['dirname'],
                    $pathinfo['filename'] . '.' . $newExtension
                );

                $result = rename($path, $newPath);

                if ($result) {
                    $this->path = basename($newPath);
                }
            }
        }

        return $result;
    }

    /**
     * Renew properties by path
     *
     * @param string $path Path
     *
     * @return boolean
     */
    protected function renewByPath($path)
    {
        $result = parent::renewByPath($path);

        if ($result) {
            $data = $this->getSystemImageData($path);

            if (is_array($data) && count($data) > 0) {
                $this->updateDimensionsSizes($data);

                $this->setMime($data['mime']);
                $hash = \Includes\Utils\FileManager::getHash($path, false, $this->includeFilenameInHash);
                if ($hash) {
                    $this->setHash($hash);
                }
            } else {
                $result = false;
            }
        }

        return $result;
    }

    public function updateDimensionsSizes(array $data = null)
    {
        if (!$data) {
            $data = $this->getSystemImageData(
                $this->getPath()
            );
        }

        if ($data) {
            $this->setWidth($data[0]);
            $this->setHeight($data[1]);
        }
    }

    public function updateMimeType(array $data = null)
    {
        if (!$data) {
            $data = $this->getSystemImageData(
                $this->getPath()
            );
        }

        if ($data) {
            $this->setMime($data['mime']);
        }
    }

    /**
     * @param $path
     *
     * @return array
     */
    protected function getSystemImageData($path)
    {
        $publicDir = 'public/';
        if (substr($path, 0, strlen($publicDir)) === $publicDir) {
            $path = substr($path, strlen($publicDir));
            $path = LC_DIR_PUBLIC . $path;
        }

        if (!file_exists($path)) {
            return [];
        }

        $result = @getimagesize($path) ?: [];

        if (
            !$result
            && strtolower(pathinfo($path, PATHINFO_EXTENSION)) === static::SVG_EXTENSION
            && (
                Request::getInstance()->svg_allowed
                || $this->isSvgAllowed()
            )
        ) {
            $request = Request::getInstance();

            $result = ($request->svgWidth || $request->svgHeight)
                ? [
                    (int) $request->svgWidth,
                    (int) $request->svgHeight
                ]
                : $this->getSvgSizes($path);

            $result['mime'] = 'image/svg+xml';
        }

        return $result;
    }

    /**
     * Load from local file
     *
     * @param string  $path       Absolute path
     * @param string  $basename   File name OPTIONAL
     * @param boolean $makeUnique True - create unique named file
     *
     * @return boolean
     */
    public function loadFromLocalFile($path, $basename = null, $makeUnique = false)
    {
        $hash = \Includes\Utils\FileManager::getHash($path, false, $this->includeFilenameInHash);

        if ($hash) {
            /** @var static $existing */
            $existing = $this->getRepository()->findOneByHash($hash);
            if ($existing && $existing->isFileExists()) {
                $path = $existing->getStoragePath();
                $basename = null;
            }
        }

        return parent::loadFromLocalFile($path, $basename, $makeUnique);
    }

    /**
     * Get name
     *
     * @return string
     */
    public function getName(): string
    {
        return $this->getStorageType() === static::STORAGE_ABSOLUTE
            ? basename($this->getPath())
            : $this->getPath();
    }

    // {{{ Resized icons

    /**
     * Get resized image URL
     *
     * @param integer $width  Width limit OPTIONAL
     * @param integer $height Height limit OPTIONAL
     * @param integer $basewidth Base Width OPTIONAL
     * @param integer $baseheight Base Height OPTIONAL
     *
     * @return array (new width, new height, URL)
     */
    public function getResizedURL($width = null, $height = null, $basewidth = null, $baseheight = null)
    {
        if (strtolower(pathinfo($this->getURL(), PATHINFO_EXTENSION)) === static::SVG_EXTENSION) {
            $result = [$this->getWidth(), $this->getHeight(), $this->getURL(), $this->getURL()];
        } elseif ($this->isUseDynamicImageResizing()) {
            $result = $this->doResize($width, $height, false, $basewidth, $baseheight);
        } else {
            $newWidth = $this->getWidth();
            $newHeight = $this->getHeight();

            if ($newWidth && $newHeight) {
                $useBaseSizes = ($basewidth && $baseheight);

                [$newWidth, $newHeight] = ImageOperator::getCroppedDimensions(
                    $newWidth,
                    $newHeight,
                    ($useBaseSizes ? $basewidth : $width),
                    ($useBaseSizes ? $baseheight : $height)
                );
            } else {
                $newWidth = $newHeight = null;
            }

            if ($this->isURL()) {
                $url = $retinaURL = $this->getURL();
            } else {
                $name = $this->getName();
                $size = ($width ?: 'x') . '.' . ($height ?: 'x');
                $path = $retinaPath = $this->getResizedPath($size, $name);

                $url = $retinaURL = $this->isResizedIconAvailable($path)
                    ? $this->getResizedPublicURL($size, $name)
                    : $this->getURL();


                if (
                    ($width * static::RETINA_RATIO) <= $this->getWidth()
                    && ($height * static::RETINA_RATIO) <= $this->getHeight()
                ) {
                    $retinaSize = ($width * static::RETINA_RATIO ?: 'x') . '.' . ($height * static::RETINA_RATIO ?: 'x');
                    $retinaPath = $this->getResizedPath($retinaSize, $name);

                    $retinaURL = $this->isResizedIconAvailable($retinaPath)
                        ? $this->getResizedPublicURL($retinaSize, $name)
                        : $url;
                }
            }

            $result = [$newWidth, $newHeight, $url, $retinaURL];
        }


        return $result;
    }

    /**
     * Resize images
     *
     * @param boolean $doRewrite
     */
    public function prepareSizes($doRewrite = false)
    {
        $this->doResizeAll($this->getAllSizes(), $doRewrite);
    }

    /**
     * Get all defined sizes
     *
     * @return array
     */
    protected function getAllSizes()
    {
        return \XLite\Logic\ImageResize\Generator::getModelImageSizes(
            get_class($this)
        );
    }

    protected function getResizeableTypes(): array
    {
        return array_filter(
            static::$types,
            static fn(string $ext): bool => (strtolower($ext) !== static::SVG_EXTENSION)
        );
    }

    /**
     * Do resize multiple
     *
     * @param array $sizes
     * @param boolean $doRewrite Rewrite flag OPTIONAL
     *
     * @return array
     */
    public function doResizeAll($sizes, $doRewrite = false)
    {
        $sizesData = $this->prepareSizesData($sizes, $doRewrite);
        $type = $this->getExtension();
        $result = [];

        if (in_array($type, $this->getResizeableTypes(), true)) {
            $operator = new ImageOperator($this);
            $result = $operator->resizeBulk($sizesData);

            foreach ($result as $size) {
                if (isset($size['tmp'])) {
                    \Includes\Utils\FileManager::write($size['path'], $size['tmp']->getBody());
                }
            }
        }

        return $result;
    }

    /**
     * Prepare sizes array
     *
     * @param array $sizes
     * @param boolean $doRewrite
     *
     * @return array
     */
    protected function prepareSizesData($sizes, $doRewrite = false)
    {
        $sizesData = [];

        foreach ($sizes as $size) {
            [$width, $height] = $size;

            if (
                $width < 0
                || $height < 0
                || ($width === 0 && $height === 0)
            ) {
                if ($width < 0 || $height < 0) {
                    $this->getLogger()->debug(
                        sprintf(
                            'To resize, image width and height must be not less than zero, got: width=%s, height=%s',
                            $width,
                            $height
                        )
                    );
                } else {
                    $this->getLogger()->debug(
                        'To resize, image width or height must be greater than zero, got: width=0, height=0'
                    );
                }

                continue;
            }

            $sizeValue = $this->checkSizeExists($width, $height, $doRewrite);

            if ($sizeValue) {
                $sizesData[] = $sizeValue;
            }

            if (
                \Includes\Utils\ConfigParser::getOptions(['images', 'generate_retina_images'])
                && ($width * static::RETINA_RATIO) <= $this->getWidth()
                && ($height * static::RETINA_RATIO) <= $this->getHeight()
            ) {
                $sizeValue = $this->checkSizeExists(
                    $width * static::RETINA_RATIO,
                    $height * static::RETINA_RATIO,
                    $doRewrite
                );
                if ($sizeValue) {
                    $sizesData[] = $sizeValue;
                }
            }
        }

        return array_unique($sizesData, SORT_REGULAR);
    }

    /**
     * @param integer $width
     * @param integer $height
     * @param boolean $doRewrite
     *
     * @return array|null
     */
    protected function checkSizeExists($width, $height, $doRewrite)
    {
        $sizeValue = null;

        [$newWidth, $newHeight] = $this->getCropDimensions($width, $height);

        $name = $this->getName();
        $size = ($width ?: 'x') . '.' . ($height ?: 'x');
        $path = $this->getResizedPath($size, $name);
        $url = $this->getResizedPublicURL($size, $name);

        if (!$this->isResizedIconAvailable($path) || $doRewrite) {
            $sizeValue = [
                'width' => $newWidth,
                'height' => $newHeight,
                'path' => $path,
                'url' => $url
            ];
        }

        return $sizeValue;
    }

    /**
     * Resize image by constraints
     *
     * @param integer $width     Width limit OPTIONAL
     * @param integer $height    Height limit OPTIONAL
     * @param boolean $doRewrite Force rewrite OPTIONAL
     * @param integer $basewidth Base width OPTIONAL
     * @param integer $baseheight Base height OPTIONAL
     *
     * @return array of [actualWidth, actualHeight, resizedPath, retinaResizedPath]
     */
    public function doResize($width = null, $height = null, $doRewrite = false, $basewidth = null, $baseheight = null)
    {
        [$newWidth, $newHeight] = $this->getCropDimensions($width, $height);

        if ($basewidth && $baseheight) {
            [$newWidth, $newHeight] = $this->getCropDimensions($basewidth, $baseheight);
        }

        $resizedPath = $retinaResizedPath = $this->resizeImage($width, $height, $doRewrite) ?: $this->getURL();

        if (
            \Includes\Utils\ConfigParser::getOptions(['images', 'generate_retina_images'])
            && ($width * static::RETINA_RATIO) <= $this->getWidth()
            && ($height * static::RETINA_RATIO) <= $this->getHeight()
        ) {
            $retinaResizedPath = $this->resizeImage(
                $width * static::RETINA_RATIO,
                $height * static::RETINA_RATIO,
                $doRewrite
            ) ?: $resizedPath;
        }

        return [$newWidth, $newHeight, $resizedPath, $retinaResizedPath];
    }

    /**
     * Perform the actual resizing and return resized path
     *
     * @param integer $width     Width limit OPTIONAL
     * @param integer $height    Height limit OPTIONAL
     * @param boolean $doRewrite Force rewrite OPTIONAL
     *
     * @return string|null Resized path or null if resizing cannot be performed
     */
    protected function resizeImage($width = null, $height = null, $doRewrite = false)
    {
        if ($this->isURL()) {
            return null;
        }

        [$newWidth, $newHeight] = $this->getCropDimensions($width, $height);

        $name = $this->getName();
        $size = ($width ?: 'x') . '.' . ($height ?: 'x');
        $path = $this->getResizedPath($size, $name);
        $url = $this->getResizedPublicURL($size, $name);
        $type = $this->getExtension();

        if (!in_array($type, static::$types, true)) {
            $url = null;
        } elseif (!$this->isResizedIconAvailable($path) || $doRewrite) {
            $result = $this->resizeIcon($newWidth, $newHeight, $path);

            if (!$result) {
                $url = null;
            }
        }

        return $url;
    }

    /**
     * Returns crop dimensions by width and height limit (preserving aspect ratio)
     *
     * @param integer $width  Width limit OPTIONAL
     * @param integer $height Height limit OPTIONAL
     *
     * @return array [newWidth, newHeight]
     */
    protected function getCropDimensions($width, $height)
    {
        return ImageOperator::getCroppedDimensions(
            $this->getWidth(),
            $this->getHeight(),
            $width,
            $height
        );
    }

    /**
     * Get resized file system path
     *
     * @param string $size Size prefix
     * @param string $name File name
     *
     * @return string
     */
    public function getResizedPath($size, $name)
    {
        return $this->getRepository()->getFileSystemCacheRoot($size) . $name;
    }

    /**
     * Get resized file public URL
     *
     * @param string $size Size prefix
     * @param string $name File name
     *
     * @return string
     */
    protected function getResizedPublicURL($size, $name)
    {
        $name = implode(LC_DS, array_map('rawurlencode', explode(LC_DS, $name)));

        return \XLite::getInstance()->getShopURL(
            $this->getRepository()->getWebCacheRoot($size) . '/' . $name,
            \XLite\Core\Request::getInstance()->isHTTPS()
        );
    }

    /**
     * Check - resized icon is available or not
     *
     * @param string $path Resized image path
     *
     * @return boolean
     */
    protected function isResizedIconAvailable($path)
    {
        return \Includes\Utils\FileManager::isFile($path) && $this->getDate() <= filemtime($path);
    }

    /**
     * Resize icon
     *
     * @param integer $width  Destination width
     * @param integer $height Destination height
     * @param string  $path   Write path
     *
     * @return array
     */
    protected function resizeIcon($width, $height, $path)
    {
        $operator = new ImageOperator($this);
        [$newWidth, $newHeight, $result] = $operator->resize($width, $height);

        return $result !== false && \Includes\Utils\FileManager::write($path, $operator->getImage()->getBody())
            ? [$newWidth, $newHeight]
            : null;
    }

    /**
     * Resize on view
     *
     * @return boolean
     */
    protected function isUseDynamicImageResizing()
    {
        return \Xlite\Core\Config::getInstance()->Performance->use_dynamic_image_resizing;
    }

    // }}}

    /**
     * Remove file
     *
     * @param string $path Path OPTIONAL
     *
     * @return void
     */
    public function removeFile($path = null)
    {
        parent::removeFile($path);

        $this->removeResizedImages($path);
    }

    /**
     * Postprocess file by temporary file
     *
     * @param \XLite\Model\TemporaryFile $temporaryFile
     */
    public function postprocessByTemporary(\XLite\Model\TemporaryFile $temporaryFile)
    {
    }

    /**
     * Prepare image before remove operation
     *
     * @param string $path Path OPTIONAL
     */
    public function removeResizedImages($path = null)
    {
        if ($this->isURL()) {
            return;
        }

        $name = $path ?: $this->getPath();
        if ($name === null) {
            return;
        }

        $sizes = $this->getAllSizes();
        $sizesData = $this->prepareSizesData($sizes, true);

        foreach ($sizesData as $size) {
            if (\Includes\Utils\FileManager::isExists($size['path'])) {
                $isDeleted = \Includes\Utils\FileManager::deleteFile($size['path']);

                if (!$isDeleted) {
                    $this->getLogger()->debug('Can\'t delete resized image ' . $size['path']);
                }
            }
        }
    }

    /**
     * Set width
     *
     * @param integer $width
     *
     * @return $this
     */
    public function setWidth($width)
    {
        $this->width = $width;
        return $this;
    }

    /**
     * Get width
     *
     * @return integer
     */
    public function getWidth()
    {
        return $this->width;
    }

    /**
     * Set height
     *
     * @param integer $height
     *
     * @return $this
     */
    public function setHeight($height)
    {
        $this->height = $height;
        return $this;
    }

    /**
     * Get height
     *
     * @return integer
     */
    public function getHeight()
    {
        return $this->height;
    }

    /**
     * Set hash
     *
     * @param string $hash
     *
     * @return $this
     */
    public function setHash($hash)
    {
        $this->hash = $hash;
        return $this;
    }

    /**
     * Get hash
     *
     * @return string
     */
    public function getHash()
    {
        return $this->hash;
    }

    /**
     * Set needProcess
     *
     * @param boolean $needProcess
     *
     * @return $this
     */
    public function setNeedProcess($needProcess)
    {
        $this->needProcess = $needProcess;
        return $this;
    }

    /**
     * Get needProcess
     *
     * @return boolean
     */
    public function getNeedProcess()
    {
        return $this->needProcess;
    }


    public function allowExtendedTypes()
    {
        static::$types = array_merge(static::$types, static::$extendedTypes);
    }

    /**
     * @return string[]
     */
    public static function getAllowedExtensions(): array
    {
        return array_unique(array_values(static::$types));
    }

    public static function getAllowedTypes(): array
    {
        return static::$types;
    }

    protected static function convertSvgSizeToPixels(string $size): int
    {
        $size = trim($size);
        $value = substr($size, 0, -2);
        $unit = substr($size, -2);

        if (is_numeric($value) && isset(static::$svgMeasurements[$unit])) {
            $pixelValue = ((float) $value) * static::$svgMeasurements[$unit];
        } elseif (is_numeric($size)) {
            $pixelValue = (float) $size;
        } else {
            $pixelValue = 0;
        }

        return (int) round($pixelValue);
    }

    /**
     * @param string $filePath
     *
     * @return array{int, int}
     */
    protected function getSvgSizes(string $filePath): array
    {
        $result = [0, 0];

        $xml = simplexml_load_file($filePath);

        if ($xml) {
            $attributes = $xml->attributes();

            $width = static::convertSvgSizeToPixels($attributes->width);
            $height = static::convertSvgSizeToPixels($attributes->height);

            // Absolute dimensions
            if ($width && $height) {
                $result = [$width, $height];
            }

            $viewBox = preg_split('/[\s,]+/', (string)$xml->attributes->viewBox);
            $viewBoxWidth = (float) ($viewBox[2] ?? 0);
            $viewBoxHeight = (float) ($viewBox[3] ?? 0);

            // Missing width/height and viewBox
            if ($viewBoxWidth > 0 && $viewBoxHeight > 0) {
                // Fixed width and viewBox
                if ($result[0] && !$result[1]) {
                    $result[1] = (int) round($result[0] / $viewBoxWidth * $viewBoxHeight);
                }

                // Fixed height and viewBox
                if ($result[1] && !$result[0]) {
                    $result[0] = (int) round($result[1] / $viewBoxHeight * $viewBoxWidth);
                }
            }
        }

        return $result;
    }

    protected function isSvgAllowed(): bool
    {
        return false;
    }
}
