<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Logic\Import\Console\Step;

use Exception;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Messenger\MessageBusInterface;
use XCart\Container;
use XCart\Messenger\Message\Import as ImportTask;
use XLite\Core\Database;
use XLite\Logic\Import\Console\Step\Base\DataStep;
use XLite\Logic\Import\Processor\AProcessor;
use XLite\Logic\Import\Processor\Products as ProductsProcessor;
use XLite\Model\TmpVar;

class Import extends DataStep
{
    /**
     * @throws Exception
     */
    public function process(): void
    {
        /** @var SymfonyStyle $io */
        $io = $this->importer->getIo();
        $progressBar = $io->createProgressBar();
        $progressBar->setFormat(static::PROGRESS_BAR_FORMAT);
        $progressBar->start();

        $processor = $this->getProcessor();
        $reader = $this->getReader($this->getFile(), $processor);
        $reader->init();
        $em = Database::getEM();

        if (
            $processor instanceof ProductsProcessor
            && $this->importer->getOptions()->useWorkers
        ) {
            /** @var MessageBusInterface $bus */
            $bus = Container::getContainer()->get('messenger.default_bus');

            $prefix = md5(mt_rand());
            $countQuery = Database::getRepo(TmpVar::class)
                ?->createPureQueryBuilder('tv')
                ->andWhere("tv.name LIKE :name")
                ->setParameter('name', $prefix . '%')
                ->andWhere("tv.value = ''")
                ->select('COUNT(tv.id)')
                ->setMaxResults(1);

            $batchNum = 1;
            $batchSize = 100;
            $count = 0;
            while ($batch = $reader->getBatch($batchSize)) {
                /** @var TmpVar $tmpVar */
                $tmpVar = Database::getRepo(TmpVar::class)
                    ?->insert(new TmpVar(['name' => $prefix . '.' . $batchNum]));
                $task = new ImportTask(
                    $tmpVar->getId(),
                    $batch,
                    get_class($processor),
                    $this->importer->getOptions()->getArrayCopy()
                );
                $bus->dispatch($task);
                $batchNum++;
                $count += count($batch);
            }

            $progressBar->setMaxSteps($count);
            $lastProgress = 0;
            while (true) {
                sleep(1);
                $res = $countQuery->getSingleScalarResult();
                $progress = $count - (int) $res * $batchSize;
                if ($progress > $lastProgress) {
                    $progressBar->setProgress($progress);
                }
                $lastProgress = $progress;

                if (!$res) {
                    $em->clear();
                    $errors = Database::getRepo(TmpVar::class)
                        ?->createPureQueryBuilder('tv')
                        ->andWhere("tv.name LIKE :name")
                        ->setParameter('name', $prefix . '%')
                        ->getResult();

                    if (!empty($errors)) {
                        foreach ($errors as $error) {
                            $message = json_decode($error->getValue(), true);
                            $io->error($message['error'] ?? '');
                        }

                        throw new \RuntimeException('Import failed');
                    }

                    break;
                }
            }
        } else {
            $n = 0;

            while ($data = $reader->getData()) {
                $n++;
                $processor->process($data, AProcessor::MODE_IMPORT);
                $progressBar->advance();

                if ($n % $processor->getChunkSize() === 0) {
                    $em->flush();
                    $em->clear();
                    $processor->clearEntitiesCaches();
                }
            }

            $em->flush();
            $em->clear();
        }

        $progressBar->finish();
    }

    public function initialize(): void
    {
        parent::initialize();

        $this->importer->getOptions()->columnsMetaData = [];
        $this->importer->getOptions()->warningsAccepted = true;
    }

    public function getFinalNote(): string
    {
        return static::t('Imported');
    }

    public function getNote(): string
    {
        return static::t('Importing data...');
    }
}
