<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Logic\Import\Console\Step;

use Symfony\Component\Console\Style\SymfonyStyle;
use XLite\Core\Database;
use XLite\Logic\Import\Console\Step\Base\DataStep;
use XLite\Logic\Import\Processor\AProcessor;

class DependenciesImport extends DataStep
{
    protected string $cacheFilename = '';

    public function process(): void
    {
        /** @var SymfonyStyle $io */
        $io = $this->importer->getIo();
        $progressBar = $io->createProgressBar();
        $progressBar->setFormat(static::PROGRESS_BAR_FORMAT);
        $progressBar->start();

        $em = Database::getEM();
        $processor = $this->getProcessor();
        $reader = $this->getReader($file = $this->getFile(), $processor);
        $this->cacheFilename = LC_DIR_VAR . 'import-tmp/' . (new \SplFileInfo($file))->getBasename('.jsonl') . '.2.jsonl';
        $reader->init();

        $n = 0;
        while ($data = $reader->getData()) {
            ++$n;
            $processor->process($data, AProcessor::MODE_DEPS_IMPORT);
            $processor->updateCacheData();

            $progressBar->advance();
            if ($n % 100 === 0) {
                $em->flush();
                if ($processor->allowTwoIterationCacheProcess($this)) {
                    $this->setCacheData($processor->shiftCacheData());
                }
                $em->clear();
            }
        }

        $this->postprocess($processor);

        $em->flush();
        if ($processor->allowTwoIterationCacheProcess($this)) {
            $this->setCacheData($processor->shiftCacheData());
        }
        $em->clear();

        $progressBar->finish();
    }

    /**
     * @throws \JsonException
     */
    protected function setCacheData(array $dataRows): void
    {
        $data = '';
        foreach ($dataRows as $dataRow) {
            $dataRow = array_map(static fn ($item) => is_callable($item) ? $item() : $item, $dataRow);
            $data .= \json_encode($dataRow, JSON_THROW_ON_ERROR) . "\n";
        }

        $this->filesystem->appendToFile($this->cacheFilename, $data);
    }

    protected function postprocess(AProcessor $processor): void
    {
        if (method_exists($processor, 'postprocessImportDeps')) {
            $processor->postprocessImportDeps();
        }
    }

    public function initialize(): void
    {
        parent::initialize();

        $this->importer->getOptions()->columnsMetaData = [];
        $this->importer->getOptions()->warningsAccepted = true;
    }

    public function getFinalNote(): string
    {
        return static::t('Dependencies imported');
    }

    public function getNote(): string
    {
        return static::t('Dependencies import...');
    }
}
