<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Logic\Import\Console\Step;

use ArrayObject;
use SplFileInfo;
use Symfony\Component\Filesystem\Filesystem;
use XLite\Core\Translation;
use XLite\Logic\Import\Console\ConsoleImportFileReaderInterface;
use XLite\Logic\Import\Console\CSVImportFileReader;
use XLite\Logic\Import\Console\JSONLImportFileReader;
use XLite\Logic\Import\ImporterInterface;
use XLite\Logic\Import\Processor\AProcessor;

abstract class AStep
{
    protected ImporterInterface $importer;

    protected Filesystem $filesystem;

    protected const PROGRESS_BAR_FORMAT = '<bg=white;fg=black>%current%</>/<info>%max%</info> <fg=black;bg=cyan>[%bar%]</> %percent:3s%% / <fg=red;options=bold,underscore>%elapsed:6s% / %memory:6s%</>';

    abstract public function process(): void;

    abstract public function getFinalNote(): string;

    abstract public function getNote(): string;

    public function __construct(ImporterInterface $importer)
    {
        $this->importer = $importer;
        $this->filesystem = new Filesystem();
    }

    public function isAllowed(): bool
    {
        return true;
    }

    public function getImporter(): ImporterInterface
    {
        return $this->importer;
    }

    public function getOptions(): ArrayObject
    {
        return $this->importer->getOptions();
    }

    public function initialize(): void
    {
    }

    public function getReader(string $file, AProcessor $processor): ConsoleImportFileReaderInterface
    {
        $extension = (new SplFileInfo($file))->getExtension();

        if ($extension === 'csv') {
            $reader = new CSVImportFileReader($file, $processor, $processor->allowJsonlCacheProcess($this));
        } elseif ($extension === 'jsonl') {
            $reader = new JSONLImportFileReader($file, $processor);
        } else {
            throw new \RuntimeException('Unknown file format.');
        }

        return $reader;
    }

    protected function getFile(): string
    {
        $file = new SplFileInfo($this->importer->getOptions()->file);

        if ($file->getExtension() === 'csv') {
            $tmpFile = LC_DIR_VAR . 'import-tmp/' . $file->getFileInfo()->getBasename('.csv') . '.jsonl';

            if (file_exists($tmpFile)) {
                $this->importer->getOptions()->file = $tmpFile;

                return $tmpFile;
            }
        }

        if ($file->getExtension() === 'jsonl') {
            $firstCacheFile = LC_DIR_VAR . 'import-tmp/' . $file->getFileInfo()->getBasename();
            $secondCacheFile = LC_DIR_VAR . 'import-tmp/' . $file->getFileInfo()->getBasename('.jsonl') . '.2.jsonl';

            if (file_exists($secondCacheFile) && file_exists($firstCacheFile)) {
                $this->filesystem->remove($firstCacheFile);
                $this->importer->getOptions()->file = $secondCacheFile;

                return $secondCacheFile;
            }
        }

        return $this->importer->getOptions()->file;
    }

    protected static function t(string $name, array $arguments = [], string $code = null, string $type = null): string
    {
        return Translation::lbl($name, $arguments, $code, $type);
    }
}
