<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Logic\Import\Console;

use ArrayObject;
use Exception;
use Symfony\Component\Console\Style\SymfonyStyle;
use XCart\Container;
use XLite\Core\Config;
use XLite\Logic\Import\Console\Processor\ProductAttributes;
use XLite\Logic\Import\Console\Step\AStep;
use XLite\Logic\Import\Console\Step\ClearCache;
use XLite\Logic\Import\Console\Step\DependenciesImport;
use XLite\Logic\Import\Console\Step\Import;
use XLite\Logic\Import\Console\Step\QuickData;
use XLite\Logic\Import\Console\Step\Verification;
use XLite\Logic\Import\Importer as BaseImporter;
use XLite\Logic\Import\ImporterInterface;
use XLite\Logic\Import\Step\CategoriesStructure;
use XLite\Logic\Import\Step\ImageResize;
use XLite\View\Import\Begin as ImportConstants;

class Importer implements ImporterInterface, ConsoleImporterInterface
{
    protected ArrayObject $options;

    protected static string $languageCode = 'en';

    protected ?SymfonyStyle $io;

    protected array $excludeSteps = [];

    public function __construct(array $options = [], ?SymfonyStyle $io = null)
    {
        $this->excludeSteps = !empty($options['skipVerification'])
            ? [Verification::class]
            : [];
        unset($options['skipVerification']);

        $options = [
                'file'               => $options['file'] ?? null,
                'charset'            => BaseImporter::DEFAULT_CHARSET,
                'delimiter'          => $options['delimiter'],
                'enclosure'          => '"',
                'ignoreFileChecking' => true,
                'columnsMetaData'    => $options['columnsMetaData'] ?? [],
                'importMode'         => ImportConstants::MODE_UPDATE_AND_CREATE,
                'processorClass'     => $options['processorClass'] ?? null,
            ] + $options;

        static::$languageCode = $options['languageCode'] ?? Config::getInstance()->General->default_admin_language;

        $this->options = new ArrayObject($options, ArrayObject::ARRAY_AS_PROPS);
        $this->io = $io;
    }

    /**
     * @throws Exception
     */
    public function getProcessorClass(): string
    {
        if ($this->getOptions()->processorClass) {
            return $this->getOptions()->processorClass;
        }

        $filename = strtolower(
            str_replace(
                '-',
                '',
                pathinfo($this->getOptions()->file)['filename']
            )
        );

        foreach (static::getProcessorList() as $processorClass) {
            $parts = explode('\\', $processorClass);

            if (str_starts_with($filename, strtolower(array_pop($parts)))) {
                $this->getOptions()->processorClass = $processorClass;

                return $processorClass;
            }
        }

        throw new \RuntimeException('There is no processor to process this file');
    }

    public static function getProcessorList()
    {
        $list = BaseImporter::getProcessorList();
        $list[] = ProductAttributes::class;

        return $list;
    }

    public static function getLanguageCode(): string
    {
        return static::$languageCode;
    }

    public static function setLanguageCode(string $code): void
    {
        static::$languageCode = $code;
    }

    public function run(): void
    {
        Container::getServiceLocator()->getCallbackManager()->setContextValue('import');

        foreach ($this->getSteps() as $k => $stepClass) {
            /** @var AStep $step */
            $step = new $stepClass($this, $k);

            if ($step->isAllowed()) {
                $this->getIo()->writeln($step->getNote());
                $step->process();
                $this->getIo()->success($step->getFinalNote());
            }
        }
    }

    protected function getSteps(): array
    {
        return array_values(array_filter([
            ClearCache::class,
            Verification::class,
            DependenciesImport::class,
            Import::class,
            QuickData::class,
            CategoriesStructure::class,
            ImageResize::class,
            ClearCache::class,
        ], fn($step) => !in_array($step, $this->excludeSteps)));
    }

    public function enableCategoriesStructureCorrection(): void
    {
        if (!isset($this->getOptions()->commonData)) {
            $this->getOptions()->commonData = [];
        }

        $this->getOptions()->commonData['correctCategoriesAllowed'] = true;
    }

    public function enableImageResize(): void
    {
        if (!isset($this->getOptions()->commonData)) {
            $this->getOptions()->commonData = [];
        }

        $this->getOptions()->commonData['imageResizeAllowed'] = true;
    }

    public function getOptions(): ArrayObject
    {
        return $this->options;
    }

    public function setOptions($options): void
    {
        $this->options = $options;
    }

    public function getIo(): ?SymfonyStyle
    {
        return $this->io;
    }
}
