<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Logic\Import\Console;

use Exception;
use Generator;
use LogicException;
use SplFileObject;
use Symfony\Component\Filesystem\Filesystem;
use XLite\Logic\Import\Processor\AProcessor;

class CSVImportFileReader implements ConsoleImportFileReaderInterface
{
    private SplFileObject $file;

    private Generator $generator;

    private AProcessor $processor;

    private Filesystem $filesystem;

    private bool $needCache;

    private string $cacheFilename;

    private ?array $lastCachedRow = null;

    private ?int $headerCount = null;

    public function __construct(string $path, AProcessor $processor, bool $needCache = false)
    {
        $this->processor = $processor;
        $this->file = new SplFileObject($path, 'r');
        $this->file->setFlags(SplFileObject::READ_CSV);
        $delimiter = $processor->getImporter()->getOptions()['delimiter'];
        $this->file->setCsvControl($delimiter);

        $this->generator = $this->initGenerator();
        $this->needCache = $needCache;

        $this->filesystem = new Filesystem();
        $this->cacheFilename = LC_DIR_VAR . 'import-tmp/' . $this->file->getFileInfo()->getBasename('.csv') . '.jsonl';
    }

    public function init(): void
    {
        $header = $this->getHeader();
        $this->headerCount = count($header);

        $this->processor->initialize();
        $this->processor->processHeaderRow([$header]);
    }

    /**
     * @throws Exception
     */
    public function getData(): ?array
    {
        $dataString = null;
        if ($rows = $this->getCollectedRows()) {
            if ($this->checkFieldsCount($rows)) {
                $dataString = $this->processor->assembleColumnsData($rows, $this->needCache);
            } else {
                throw new Exception($this->processor::getMessages()['CMN-INVALID-ROW-LENGTH'] ?? 'Log code is unknown');
            }
        }

        if ($dataString && $this->needCache) {
            $this->setCacheString($dataString);
        }

        return $dataString;
    }

    protected function checkFieldsCount(array $rows): bool
    {
        return (bool) array_filter(
            $rows,
            fn (array $row): bool => count($row) === $this->headerCount
        );
    }

    protected function setCacheString(array $dataRow): void
    {
        $dataRow = \json_encode($dataRow);
        $this->filesystem->appendToFile($this->cacheFilename, $dataRow . "\n");
    }

    /**
     * @throws LogicException
     */
    public function getHeader(): array
    {
        if ($this->generator->key() !== 0) {
            throw new LogicException('Too late to read the header');
        }

        return $this->getRow();
    }

    public function getCollectedRows(): ?array
    {
        $rows = [];

        if ($this->lastCachedRow) {
            $mainRow = $this->lastCachedRow;
            $this->lastCachedRow = null;
        } else {
            $mainRow = $this->getRow();

            if (is_null($mainRow)) {
                return null;
            }
        }

        $rows[] = $mainRow;

        while ($row = $this->getRow()) {
            if ($this->isSubrow($mainRow, $row)) {
                $rows[] = $row;
            } else {
                $this->lastCachedRow = $row;
                break;
            }
        }

        return $rows;
    }

    public function getBatch(int $size = 100): array
    {
        $batch = [];

        $i = 0;
        while ($i < $size) {
            $data = $this->getData();
            if (!$data) {
                break;
            }

            $batch[] = $data;
            $i++;
        }

        return $batch;
    }

    private function getRow(): ?array
    {
        if (!$this->generator->valid()) {
            return null;
        }

        $row = $this->generator->current();
        $this->generator->next();

        if (empty(array_filter($row))) {
            return $this->getRow();
        }

        return $row;
    }

    private function isSubrow(array $mainRow, array $row): bool
    {
        $result = true;
        $empty = true;

        foreach ($this->processor->getKeyColumns() as $column) {
            $empty = $empty && !$this->processor->getColumnValue($column, $mainRow);

            if ($this->processor->getColumnValue($column, $mainRow) !== $this->processor->getColumnValue($column, $row)) {
                $result = false;
                break;
            }
        }

        return $empty ? false : $result;
    }

    private function initGenerator(): Generator
    {
        foreach ($this->file as $row) {
            yield $row;
        }
    }
}
