<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Core\RemoteResource;

use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\Exception\HttpExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use XLite\Core\HttpClient\Exception\ResponseInternalException;
use Exception;

abstract class AURL implements IURL
{
    /**
     * @var string
     */
    protected $url;

    /**
     * @var array
     */
    protected $headers;

    /**
     * @param string $url
     *
     * @return boolean
     */
    protected static function isURL($url)
    {
        return (bool) filter_var($url, FILTER_VALIDATE_URL);
    }

    /**
     * @param string $url
     *
     * @return string
     */
    abstract public function convertURL($url);

    public function __construct($url)
    {
        $this->url = $this->convertURL($url);
    }

    /**
     * @return string
     */
    public function getName()
    {
        return basename(parse_url($this->getURL(), \PHP_URL_PATH));
    }

    /**
     * @return boolean
     */
    public function isAvailable()
    {
        try {
            $headers       = $this->getHeaders();
            $contentLength = $headers['content-length'][0] ?? null;
        } catch (Exception) {
            return false;
        }

        if ($contentLength === null) {
            return false;
        }

        if (str_contains($contentLength, ',')) {
            $contentArray = explode(',', $contentLength);
            return (int)end($contentArray) > 0;
        }

        return (int)$contentLength > 0;
    }

    /**
     * @return string
     */
    public function getURL()
    {
        return $this->url;
    }

    /**
     * @param string $url
     */
    public function setURL($url)
    {
        $this->url = $url;
    }

    public function getHeaders()
    {
        if ($this->headers === null) {
            try {
                $this->headers = $this->getHeadHeaders();
            } catch (HttpExceptionInterface | TransportExceptionInterface) {
            }

            try {
                $this->headers = $this->getGetHeaders();
            } catch (HttpExceptionInterface | TransportExceptionInterface) {
            }
        }

        if ($this->headers) {
            return $this->headers;
        }

        throw new Exception();
    }

    /**
     * @return array
     * @throws HttpExceptionInterface|TransportExceptionInterface
     */
    protected function getHeadHeaders()
    {
        $response = $this->getHttpClient()->request(
            'HEAD',
            $this->getURL()
        );
        $headers = $response->getHeaders();
        if ($response->getStatusCode() !== 200) {
            throw new ResponseInternalException($response);
        }

        return $headers;
    }

    /**
     * @return array
     * @throws HttpExceptionInterface|TransportExceptionInterface
     */
    protected function getGetHeaders()
    {
        $response = $this->getHttpClient()->request(
            'GET',
            $this->getURL(),
            [
                'headers' => [
                    'Range' => 'bytes=0-0'
                ]
            ]
        );
        $headers = $response->getHeaders();
        if (!in_array($response->getStatusCode(), [200, 206], true)) {
            throw new ResponseInternalException($response);
        }

        return $headers;
    }

    protected function getHttpClient(): HttpClientInterface
    {
        return HttpClient::create();
    }
}
