<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Core;

use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\RedirectionExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\ServerExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use XLite\Core\HttpClient\Helper as HttpClientHelper;
use XLite\Core\HttpClient\Exception\ResponseInternalExceptionInterface;
use XLite\Core\HttpClient\Exception\ResponseInternalException;
use XLite\Model\Profile;

/**
 * Common operations repository
 */
class Operator extends \XLite\Base\Singleton
{
    /**
     * Token characters list
     *
     * @var array
     */
    protected $chars = [
        '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
        'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j',
        'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
        'u', 'v', 'w', 'x', 'y', 'z', 'A', 'B', 'C', 'D',
        'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N',
        'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
        'Y', 'Z', '_', '|', '!', '^', '*', '-', '~',
    ];

    /**
     * Redirect
     *
     * @param string  $location URL
     * @param integer $code     Operation code OPTIONAL
     *
     * @return void
     */
    public static function redirect($location, $code = 302)
    {
        static::setHeaderLocation($location, $code);
        static::finish();
    }

    protected static function getHttpClient(): HttpClientInterface
    {
        return HttpClient::create();
    }

    protected static function getHttpClientHelper(): HttpClientHelper
    {
        return new HttpClientHelper(static::getHttpClient());
    }

    /**
     * Try to make HEAD request and check if resource is available. Returns headers if request is successful.
     *
     * @param string $url URL
     *
     * @return array|null
     */
    public static function checkURLAvailability($url)
    {
        $url = ltrim($url, '/');
        try {
            $response = static::getHttpClient()->request(
                'HEAD',
                $url,
                static::getRequestOptionsForCheckUrlAvailability()
            );
            if ($response->getStatusCode() !== 200) {
                $response = static::getHttpClient()->request(
                    'GET',
                    $url,
                    static::getRequestOptionsForCheckUrlAvailability()
                );
            }
            if ($response->getStatusCode() === 200) {
                return $response->getHeaders();
            }
        } catch (TransportExceptionInterface) {
        }

        return null;
    }

    protected static function getRequestOptionsForCheckUrlAvailability(): array
    {
        return [];
    }

    /**
     * @throws TransportExceptionInterface   When a network error occurs
     * @throws RedirectionExceptionInterface On a 3xx when $throw is true and the "max_redirects" option has been reached
     * @throws ClientExceptionInterface      On a 4xx when $throw is true
     * @throws ServerExceptionInterface      On a 5xx when $throw is true
     * @throws ResponseInternalExceptionInterface
     * @throws IOExceptionInterface
     */
    public static function writeURLContentsToFile(string $url, string $file): void
    {
        $response = static::getHttpClient()->request(
            'GET',
            $url,
            static::getRequestOptionsForWriteURLContentsToFile()
        );
        static::getHttpClientHelper()->uploadResponseContent($response, $file);
        if ($response->getStatusCode() !== 200) {
            throw new ResponseInternalException($response);
        }
    }

    protected static function getRequestOptionsForWriteURLContentsToFile(): array
    {
        return [];
    }

    /**
     * Get URL content
     *
     * @param string $url URL
     *
     * @return string|null
     */
    public static function getURLContent($url)
    {
        try {
            $response = static::getHttpClient()->request(
                'GET',
                $url
            );
            if ($response->getStatusCode() === 200) {
                return $response->getContent();
            }
        } catch (TransportExceptionInterface) {
        }

        return null;
    }

    /**
     * Calculate pagination info
     *
     * @param integer $count Items count
     * @param integer $page  Current page index OPTIONAL
     * @param integer $limit Page length limit OPTIONAL
     *
     * @return array (pages count + current page number)
     */
    public static function calculatePagination($count, $page = 1, $limit = 20)
    {
        $count = max(0, (int) $count);
        $limit = max(0, (int) $limit);

        if ($limit == 0 && $count) {
            $pages = 1;
        } else {
            $pages = $count == 0 ? 0 : ceil($count / $limit);
        }

        $page = min($pages, max(1, (int) $page));

        return [$pages, $page];
    }

    /**
     * setHeaderLocation
     *
     * @param string  $location URL
     * @param integer $code     Operation code OPTIONAL
     *
     * @return void
     */
    protected static function setHeaderLocation($location, $code = 302)
    {
        $location = \Includes\Utils\Converter::removeCRLF($location);
        if (empty($location)) {
            $location = '/';
        }

        if (headers_sent()) {
            // HTML meta tags-based redirect
            echo (
                '<script>' . "\n"
                . '<!--' . "\n"
                . 'self.location=\'' . $location . '\';' . "\n"
                . '-->' . "\n"
                . '</script>' . "\n"
                . '<noscript><a href="' . $location . '">Click here to redirect</a></noscript><br /><br />'
            );
        } elseif (\XLite\Core\Request::getInstance()->isAJAX() && $code == 200) {
            // AJAX-based redirect
            \XLite::getInstance()->setStatusCode($code);
            \XLite::getInstance()->addHeader('AJAX-Location', $location, true);
        } else {
            // HTTP-based redirect
            \XLite::getInstance()->setStatusCode($code);
            \XLite::getInstance()->addHeader('Location', $location, true);
        }
    }

    /**
     * finish
     *
     * @return void
     */
    protected static function finish()
    {
        $session = \XLite\Core\Session::getInstance();

        /** @var Profile $profile */
        $profile = Database::getRepo('XLite\Model\Profile')->find($session->get('profile_id'));

        if ($profile) {
            $session->set('salt', $profile->getSalt());
        }

        \XLite::getInstance()->sendResponse();
        \XLite::getInstance()->runPostRequestActions();
        exit(0);
    }

    /**
     * Get class name as keys list
     *
     * @param string|object $class Class name or object
     *
     * @return array
     */
    public function getClassNameAsKeys($class)
    {
        if (is_object($class)) {
            $class = get_class($class);
        }

        $parts = explode('\\', $class);

        if ($parts[0] === 'XLite') {
            $parts = array_slice($parts, 2);
        } else {
            unset($parts[2]);
        }

        return array_map('strtolower', array_values($parts));
    }

    /**
     * Generate token
     *
     * @param integer $length Length OPTIONAL
     * @param array   $chars  Characters book OPTIONAL
     *
     * @return string
     */
    public function generateToken($length = 32, array $chars = [])
    {
        if (!$chars) {
            $chars = $this->chars;
        }

        $limit = count($chars) - 1;
        $x = explode('.', uniqid('', true));
        mt_srand((int)microtime(true) + (int) hexdec($x[0]) + $x[1]);

        $password = '';
        for ($i = 0; $length > $i; $i++) {
            $password .= $chars[mt_rand(0, $limit)];
        }

        return $password;
    }
}
