<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Controller\Admin;

use XLite\Core\Auth;
use XLite\Core\OrderHistory;

/**
 * Order page controller
 */
class Order extends \XLite\Controller\Admin\AAdmin
{
    /**
     * Controller parameters
     *
     * @var array
     */
    protected $params = ['target', 'order_id', 'order_number', 'page'];

    /**
     * Order (local cache)
     *
     * @var \XLite\Model\Order
     */
    protected $order;

    /**
     * Modifiers
     *
     * @var array
     */
    protected $modifiers;

    /**
     * Order changes
     *
     * @var array
     */
    protected static $changes = [];

    /**
     * Temporary orders list
     *
     * @var \XLite\Model\Order[]
     */
    protected static $tmpOrders;

    /**
     * Run-time flag: true if order items changes should affect stock
     *
     * @var boolean
     */
    protected static $isNeedProcessStock = false;

    /**
     * Cache of orders (for print invoice page)
     *
     * @var \XLite\Model\Order[]
     */
    protected $orders;

    /**
     * Return variable $isNeedProcessStock
     *
     * @return boolean
     */
    public static function isNeedProcessStock()
    {
        return static::$isNeedProcessStock;
    }

    /**
     * Define the actions with no secure token
     *
     * @return array
     */
    public static function defineFreeFormIdActions()
    {
        return array_merge(
            parent::defineFreeFormIdActions(),
            ['calculate_price', 'recalculate_shipping']
        );
    }

    // {{{ Temporary order related methods

    /**
     * Get temporary order (clone) for order with specified order ID
     *
     * @param integer $orderId Current order ID
     * @param boolean $force   Force temporary order creation if it doesn't exist OPTIONAL
     *
     * @return \XLite\Model\Order
     */
    public static function getTemporaryOrder($orderId, $force = true)
    {
        $result = null;

        $orderId = (int) $orderId;

        if (0 < $orderId) {
            $result = static::getTemporaryOrderFromCache($orderId)
                ?: ($force ? static::createTemporaryOrder($orderId) : null);
        }

        return $result;
    }

    /**
     * Get temporary order from cache
     *
     * @param integer $orderId Order ID
     *
     * @return \XLite\Model\Order
     */
    public static function getTemporaryOrderData($orderId = null)
    {
        $result = null;

        if (static::$tmpOrders === null) {
            // Initialize $tmpOrders
            static::$tmpOrders = [];
        }

        if ($orderId && isset(static::$tmpOrders[$orderId])) {
            // Get specific temporary order data from cache
            $result = static::$tmpOrders[$orderId];
        } else {
            $result = static::$tmpOrders;
        }

        return $result;
    }

    /**
     * Get temporary order from cache
     *
     * @param integer $orderId Order ID
     *
     * @return \XLite\Model\Order
     */
    protected static function getTemporaryOrderFromCache($orderId)
    {
        $result = null;

        if (static::$tmpOrders === null) {
            // Initialize $tmpOrders
            static::$tmpOrders = [];
        }

        if (isset(static::$tmpOrders[$orderId])) {
            // Get order from cache

            if (!(static::$tmpOrders[$orderId]['order'] instanceof \XLite\Model\Order)) {
                static::$tmpOrders[$orderId]['order'] = \XLite\Core\Database::getRepo('XLite\Model\Order')
                    ->find((int) static::$tmpOrders[$orderId]['order']);
            }

            $result = static::$tmpOrders[$orderId]['order'];
        }

        return $result;
    }

    /**
     * Create temporary order
     *
     * @param integer $orderId Order ID
     *
     * @return \XLite\Model\Order
     */
    protected static function createTemporaryOrder($orderId)
    {
        $result = null;

        // Get current order by orderId
        $order = \XLite\Core\Database::getRepo('XLite\Model\Order')->find($orderId);

        if ($order) {
            // Get temporary order as a clone
            $newOrder = $order->cloneOrderAsTemporary();

            if ($newOrder) {
                // Save cloned order in the database
                \XLite\Core\Database::getEM()->persist($newOrder);
                \XLite\Core\Database::getEM()->flush();

                // Transform temporary order to the cart object
                $newOrder->markAsCart();

                // Result is a cloned order
                $result = $newOrder;
                static::$tmpOrders[$orderId]['order'] = $newOrder;

                // Prepare the correspondence table of order items

                $itemIds = [];

                foreach ($newOrder->getItems() as $i) {
                    $itemIds[$i->getKey()] = $i->getItemId();
                }

                foreach ($order->getItems() as $item) {
                    $key = $item->getKey();
                    if (!empty($itemIds[$key])) {
                        static::$tmpOrders[$orderId]['items'][$item->getItemId()] = $itemIds[$key];
                    }
                }
            } else {
                \XLite\Core\TopMessage::addError('Cannot create temporary order for modification');
            }
        }

        return $result;
    }

    // }}}

    // {{{ Order changes processing

    /**
     * Set order changes
     *
     * @param string $name     Order property name
     * @param mixed  $newValue New property value
     * @param mixed  $oldValue Old property value
     */
    public static function setOrderChanges($name, $newValue, $oldValue = null)
    {
        static::$changes[(string)$name] = [
            'old' => $oldValue,
            'new' => $newValue,
        ];
    }

    /**
     * Return requested changes for the order
     *
     * @return array
     */
    protected function getOrderChanges()
    {
        $changes = [];

        foreach (static::$changes as $key => $data) {
            $names = explode(':', $key, 2);

            $name = static::getFieldHumanReadableName($names[0]);
            $subname = $names[1] ?? null;

            if ($subname) {
                $subname = static::getFieldHumanReadableName($subname);
                $changes[$name][$subname] = $data;
            } else {
                $changes[$name] = $data;
            }
        }

        return $changes;
    }

    /**
     * Get human readable field name
     *
     * @param string $name Field service name
     *
     * @return string
     */
    protected static function getFieldHumanReadableName($name)
    {
        $names = static::getFieldHumanReadableNames();

        return $names[$name] ?? $name;
    }

    /**
     * Get human readable field names
     *
     * @return array
     */
    protected static function getFieldHumanReadableNames()
    {
        return [
            'billingAddress'  => 'Billing address',
            'shippingAddress' => 'Shipping address',
            'shippingId'      => 'Shipping method',
            'paymentMethod'   => 'Payment method',
            'adminNote'       => 'Staff note',
            'customerNote'    => 'Customer note',
            'SHIPPING'        => 'Shipping cost',
            'firstname'       => 'First name',
            'lastname'        => 'Last name',
            'street'          => 'Address',
            'city'            => 'City',
            'zipcode'         => 'Zip code',
            'phone'           => 'Phone',
        ];
    }

    // }}}

    /**
     * Check ACL permissions
     *
     * @return boolean
     */
    public function checkACL()
    {
        return parent::checkACL() || \XLite\Core\Auth::getInstance()->isPermissionAllowed('ROLE_MANAGE_ORDERS');
    }

    public function handleRequest()
    {
        $request = \XLite\Core\Request::getInstance();

        if (
            $request->action
            && $request->action !== 'update'
        ) {
            $order = $this->getOrder();

            if ($order !== null) {
                $allowedTransactions = $order->getAllowedPaymentActions();

                if (isset($allowedTransactions[$request->action])) {
                    $request->transactionType = $request->action;
                    $request->action = 'PaymentTransaction';
                    $request->setRequestMethod('POST');
                }
            }
        }

        // Set ignoreLongCalculations mode for shipping rates gathering
        foreach (\XLite\Model\Shipping::getProcessors() as $processor) {
            if (
                !($processor instanceof \XLite\Model\Shipping\Processor\Offline)
                && $processor->isConfigured()
            ) {
                \XLite\Model\Shipping::setIgnoreLongCalculationsMode(true);

                break;
            }
        }

        parent::handleRequest();
    }

    /**
     * Check controller visibility
     *
     * @return boolean
     */
    protected function isVisible()
    {
        return parent::isVisible() && $this->getOrders();
    }

    /**
     * Check if current page is accessible
     *
     * @return boolean
     */
    public function checkAccess()
    {
        $orders = $this->getOrders();

        return parent::checkAccess()
            && (isset($orders[0])
                && $orders[0]->getProfile());
    }

    /**
     * Return the current page title (for the content area)
     *
     * @return string
     */
    public function getTitle()
    {
        if (\XLite\Core\Request::getInstance()->mode === 'invoice') {
            $result = (
                \XLite\Core\Config::getInstance()->Company->company_name
                ? \XLite\Core\Config::getInstance()->Company->company_name . ': '
                : ''
            ) . static::t('Invoice');
        } elseif (\XLite\Core\Request::getInstance()->mode === 'packing_slip') {
            $result = (
                \XLite\Core\Config::getInstance()->Company->company_name
                    ? \XLite\Core\Config::getInstance()->Company->company_name . ': '
                    : ''
                ) . static::t('Packing slip');
        } elseif (ltrim((string) \XLite\Core\Request::getInstance()->widget, '\\') === 'XLite\View\Address\OrderModify') {
            $result = static::t('Customer information');
        } elseif (ltrim((string) \XLite\Core\Request::getInstance()->widget, '\\') === 'XLite\View\SelectAddressOrder') {
            $result = static::t('Pick address from address book');
        } elseif (ltrim((string) \XLite\Core\Request::getInstance()->widget, '\\') === 'XLite\View\PaymentMethodData') {
            $paymentMethod = null;
            if (intval(\XLite\Core\Request::getInstance()->transaction_id)) {
                $transaction = \XLite\Core\Database::getRepo('XLite\Model\Payment\Transaction')
                    ->find(\XLite\Core\Request::getInstance()->transaction_id);
                if ($transaction) {
                    $paymentMethod = $transaction->getPaymentMethod()
                        ? $transaction->getPaymentMethod()->getName()
                        : $transaction->getMethodName();
                }
            }

            $result = $paymentMethod ?: static::t('Payment method data');
        } elseif ($this->getOrder()) {
            $result = static::t('Order X', ['id' => $this->getOrder()->getOrderNumber()]);
        } else {
            $result = parent::getTitle();
        }

        return $result;
    }

    /**
     * Get order
     *
     * @return \XLite\Model\Order
     */
    public function getOrder()
    {
        if ($this->order === null) {
            $order = null;
            if (\XLite\Core\Request::getInstance()->order_id) {
                $order = \XLite\Core\Database::getRepo('XLite\Model\Order')
                    ->find((int) \XLite\Core\Request::getInstance()->order_id);
            } elseif (\XLite\Core\Request::getInstance()->order_number) {
                $order = \XLite\Core\Database::getRepo('XLite\Model\Order')
                    ->findOneByOrderNumber(\XLite\Core\Request::getInstance()->order_number);
            }

            $this->order = $order instanceof \XLite\Model\Cart
                ? null
                : $order;
        }

        return $this->order;
    }

    /**
     * Get list of orders (to print invoices)
     *
     * @return array
     */
    public function getOrders()
    {
        if ($this->orders === null) {
            $result = [];

            if (\XLite\Core\Request::getInstance()->order_ids) {
                $orderIds = explode(',', \XLite\Core\Request::getInstance()->order_ids);

                foreach ($orderIds as $orderId) {
                    $orderId = trim($orderId);
                    $order = \XLite\Core\Database::getRepo('XLite\Model\Order')->find((int) $orderId);
                    if ($order) {
                        $result[] = $order;
                    }
                }
            } elseif ($this->getOrder()) {
                $result[] = $this->getOrder();
            }

            $this->orders = $result;
        }

        return $this->orders;
    }

    /**
     * Return trus if page break is required after invoice page printing
     *
     * @param integer $index Index of order in the array of orders
     *
     * @return boolean
     */
    public function hasPageBreak($index)
    {
        return $index + 1 < count($this->orders);
    }

    /**
     *
     * @return boolean
     */
    public function isAdminNoteVisible()
    {
        return true;
    }

    /**
     *
     * @return boolean
     */
    public function isBillingAddressVisible()
    {
        return $this->getOrder()->getProfile()
            && $this->getOrder()->getProfile()->getBillingAddress();
    }

    /**
     *
     * @return boolean
     */
    public function isShippingAddressVisible()
    {
        return $this->getOrder()->getProfile()
            && $this->getOrder()->getProfile()->getShippingAddress();
    }

    /**
     *
     * @return boolean
     */
    public function isOrderItemsVisible()
    {
        return true;
    }

    /**
     * Check - item price is controlled by server or not
     *
     * @param \XLite\Model\OrderItem $item Order item
     *
     * @return boolean
     */
    public function isPriceControlledServer(\XLite\Model\OrderItem $item)
    {
        return $item->isPriceControlledServer();
    }

    /**
     * Get address
     *
     * @return \XLite\Model\Address
     */
    public function getAddress()
    {
        $id = \XLite\Core\Request::getInstance()->addressId;

        return $id ? \XLite\Core\Database::getRepo('\XLite\Model\Address')->find($id) : null;
    }

    /**
     * Check - surcharge is controlled automatically or not
     *
     * @param array $surcharge Surcharge
     *
     * @return boolean
     */
    public function isAutoSurcharge(array $surcharge)
    {
        $data = \XLite\Core\Request::getInstance()->auto;

        return !empty($data)
            && !empty($data['surcharges'])
            && !empty($data['surcharges'][$surcharge['code']])
            && !empty($data['surcharges'][$surcharge['code']]['value']);
    }

    /**
     * Return true if order can be edited
     *
     * @return boolean
     */
    public function isOrderEditable()
    {
        return true;
    }

    /**
     * getRequestData
     * TODO: to remove
     *
     * @return array
     */
    protected function getRequestData()
    {
        return \Includes\Utils\ArrayManager::filterByKeys(
            \XLite\Core\Request::getInstance()->getData(),
            ['paymentStatus', 'shippingStatus']
        );
    }

    /**
     * Recalculate shipping rates of the source order
     */
    protected function doActionRecalculateShipping()
    {
        if ($this->isOrderEditable()) {
            // Set ignoreLongCalculations mode for shipping rates gathering
            \XLite\Model\Shipping::setIgnoreLongCalculationsMode(false);

            // Get source order
            $order = $this->getOrder();

            if (\XLite\Core\Request::getInstance()->isAJAX()) {
                $this->displayRecalculateShippingData($order);
                $this->restoreFormId();
            }
        }
    }

    /**
     * Display recalculate shipping data
     *
     * @param \XLite\Model\Order $order Order
     */
    protected function displayRecalculateShippingData(\XLite\Model\Order $order)
    {
        \XLite\Core\Event::recalculateShipping($this->assembleRecalculateShippingEvent($order));
    }

    /**
     * Assemble recalculate shipping event
     *
     * @param \XLite\Model\Order $order Order
     *
     * @return array
     */
    protected function assembleRecalculateShippingEvent(\XLite\Model\Order $order)
    {
        $result = [
            'options' => [],
        ];

        $modifier = $order->getModifier(\XLite\Model\Base\Surcharge::TYPE_SHIPPING, 'SHIPPING');
        $modifier->setMode(\XLite\Logic\Order\Modifier\AModifier::MODE_CART);

        foreach ($modifier->getRates() as $rate) {
            $result['options'][$rate->getMethod()->getMethodId()] = [
                'name'     => html_entity_decode(
                    strip_tags($rate->getMethod()->getName()),
                    ENT_COMPAT,
                    'UTF-8'
                ),
                'fullName' => html_entity_decode(
                    $rate->getMethod()->getName(),
                    ENT_COMPAT,
                    'UTF-8'
                ),
            ];
        }

        if (
            $order->getShippingId() > 0
            && !in_array($order->getShippingId(), array_keys($result['options']))
        ) {
            $result['options'][$order->getShippingId()] = [
                'name'     => $order->getShippingMethodName(),
                'fullName' => $order->getShippingMethodName(),
            ];
        }

        return $result;
    }

    /**
     * Recalculate order
     */
    protected function doActionRecalculate()
    {
        if ($this->isOrderEditable()) {
            // Set ignoreLongCalculations mode for shipping rates gathering
            \XLite\Model\Shipping::setIgnoreLongCalculationsMode(false);

            // Initialize temprorary order
            $order = static::getTemporaryOrder($this->getOrder()->getOrderId(), true);

            // Update order items list
            $this->updateOrderItems($order);

            // Perform 'recalculate' action via model form
            $this->getOrderForm()->performAction('recalculate');

            if (\XLite\Core\Request::getInstance()->isAJAX()) {
                $this->displayRecalculateData($order);
                $this->displayRecalculateShippingData($order);
                $this->restoreFormId();
                $this->removeTemporaryOrder($order);
            }
        }
    }

    /**
     * Update order items list
     *
     * @param \XLite\Model\Order $order Order object
     */
    protected function updateOrderItems($order)
    {
        $list = new \XLite\View\ItemsList\Model\OrderItem(
            [
                'order' => $order,
            ]
        );

        $list->processQuick();

        $order->calculateInitialValues();
    }

    /**
     * Update product sales
     *
     * @param array $before Array of saled product in order
     * @param \XLite\Model\Order $order Order
     */
    public function updateProductSales($before, $order)
    {
        foreach ($order->getItems() as $item) {
            if (isset($before[$item->getID()])) {
                if ($before[$item->getId()]['amount'] != $item->getAmount()) {
                    $diff = $item->getAmount() - $before[$item->getId()]['amount'];
                    if ($item->getObject()) {
                        $item->getObject()->setSales($item->getObject()->getSales() + $diff);
                    }
                }
                unset($before[$item->getID()]);
            } else {
                if ($item->getObject()) {
                    $item->getObject()->setSales($item->getObject()->getSales() + $item->getAmount());
                }
            }
        }
        foreach ($before as $item) {
            if ($item['item']->getObject()) {
                $item['item']->getObject()->setSales($item['item']->getObject()->getSales() - $item['amount']);
            }
        }
    }

    /**
     * Display recalculate data
     *
     * @param \XLite\Model\Order $order Order
     */
    protected function displayRecalculateData(\XLite\Model\Order $order)
    {
        if ($this->needSendRecalculateEvent($order)) {
            \XLite\Core\Event::recalculateOrder($this->assembleRecalculateOrderEvent($order));
        }
    }

    /**
     * Remove temporary order
     *
     * @param \XLite\Model\Order $order Order
     */
    protected function removeTemporaryOrder(\XLite\Model\Order $order)
    {
        $origOrderId = null;

        // Search for index of data in temporary orders static cache
        if (is_array(static::$tmpOrders)) {
            foreach (static::$tmpOrders as $id => $data) {
                if ($id == $order->getOrderId()) {
                    $origOrderId = $id;
                    break;
                }
            }
        }

        \XLite\Core\Database::getEM()->remove($order->getProfile());

        // Remove temporary order
        \XLite\Core\Database::getEM()->remove($order);
        // \XLite\Core\Database::getEM()->flush();

        // Unset data in static cache
        if ($origOrderId) {
            unset(static::$tmpOrders[$origOrderId]);
        }
    }

    /**
     * Check - need send recalculate event or not
     *
     * @param \XLite\Model\Order $order Order
     *
     * @return boolean
     */
    protected function needSendRecalculateEvent(\XLite\Model\Order $order)
    {
        return true;
    }

    /**
     * Assemble recalculate order event
     *
     * @param \XLite\Model\Order $order Order
     *
     * @return array
     */
    protected function assembleRecalculateOrderEvent(\XLite\Model\Order $order)
    {
        $result =  [
            'subtotal'  => $order->getSubtotal(),
            'total'     => $order->getTotal(),
            'modifiers' => [],
        ];

        foreach ($this->getSurchargeTotals(true) as $surcharge) {
            $result['modifiers'][$surcharge['code']] = abs($surcharge['cost']);
        }

        if ($this->isForbiddenOrderChanges($order)) {
            $result['forbidden'] = true;
        }

        return $result;
    }

    /**
     * Return true if order changes can not be saved
     *
     * @param \XLite\Model\Order $order Order entity
     *
     * @return boolean
     */
    protected function isForbiddenOrderChanges(\XLite\Model\Order $order)
    {
        $result = false;

        if (0 > $order->getTotal()) {
            $result = true;
            \XLite\Core\TopMessage::addError('Order changes cannot be saved due to negative total value');
        }

        return $result;
    }

    protected function doActionUpdate()
    {
        $order = $this->getOrder();

        static::$isNeedProcessStock = true;

        // Set this flag to prevent duplicate 'Order changed' email notifications
        $this->getOrder()->setIgnoreCustomerNotifications($this->getIgnoreCustomerNotificationFlag());
        $oldSentFlagValue = $this->getOrder()->setIsNotificationSent(true);

        if ($this->isOrderEditable()) {
            // Update order items list
            $before = $this->getOrderItemsAndAmountBeforeUpdate($order);
            $this->updateOrderItems($order);
            $this->updateProductSales($before, $order);

            // Update common order form fields
            $this->updateCommonForm();
        }

        $this->getOrder()->setIsNotificationSent($oldSentFlagValue);

        $this->updatePaymentMethods();

        // Process order tracking
        $this->updateTracking();

        // Process change order statuses
        $this->updateOrderStatus();

        // Update staff note
        $this->updateAdminNotes();

        // Update customer note (visible on invoice)
        $this->updateCustomerNotes();

        if ($this->isOrderChanged()) {
            if ($this->getOrderChanges()) {
                \XLite\Core\OrderHistory::getInstance()
                    ->registerGlobalOrderChanges($order->getOrderId(), $this->getOrderChanges());
            }

            $this->sendOrderChangeNotification();
        }

        \XLite\Core\Database::getEM()->flush();
    }

    protected function isOrderChanged()
    {
        return (bool)$this->getOrderChanges();
    }

    /**
     * Get order items amount and model before update
     *
     * @param \XLite\Model\Order $order Order
     *
     * @return array
     */
    public function getOrderItemsAndAmountBeforeUpdate($order)
    {
        foreach ($order->getItems() as $item) {
            $array[$item->getID()]['amount'] = $item->getAmount();
            $array[$item->getID()]['item']   = $item;
        }

        return $array;
    }

    protected function updateCommonForm()
    {
        $this->getOrderForm()->performAction('save');
    }

    protected function updateOrderStatus()
    {
        $data = $this->getRequestData();
        $order = $this->getOrder();

        $updateRecent = false;
        foreach (['paymentStatus', 'shippingStatus'] as $status) {
            $method = 'get' . \Includes\Utils\Converter::convertToUpperCamelCase($status);
            // Call assembled $method: getPaymentStatus() or getShippingStatus()
            $oldStatus = $order->$method();
            $oldStatusId = $oldStatus ? $oldStatus->getId() : null;
            if (!empty($data[$status]) && (!$oldStatus || $oldStatusId != $data[$status])) {
                $updateRecent = true;
            }
        }

        if ($updateRecent) {
            $data['recent'] = 0;
        }

        \XLite\Core\Database::getRepo('\XLite\Model\Order')->updateById(
            $order->getOrderId(),
            $data
        );
    }

    protected function updateTracking()
    {
        $numbers = $this->prepareTrackingData($this->getOrder()->getTrackingNumbers()->toArray());

        $list = new \XLite\View\ItemsList\Model\OrderTrackingNumber(
            [
                \XLite\View\ItemsList\Model\OrderTrackingNumber::PARAM_ORDER_ID => $this->getOrder()->getOrderId(),
            ]
        );
        $list->processQuick();

        $this->processTrackingInfoChanges(
            $numbers,
            $this->prepareTrackingData($this->getOrder()->getTrackingNumbers()->toArray())
        );
    }

    /**
     * @param \XLite\Model\OrderTrackingNumber[] $numbers
     *
     * @return array
     */
    protected function prepareTrackingData(array $numbers)
    {
        return array_combine(
            array_map(static function (\XLite\Model\OrderTrackingNumber $number) {
                return $number->getTrackingId();
            }, $numbers),
            array_map(static function (\XLite\Model\OrderTrackingNumber $number) {
                return $number->getValue();
            }, $numbers)
        );
    }

    /**
     * @param array $old
     * @param array $new
     */
    protected function processTrackingInfoChanges(array $old, array $new)
    {
        $deleted = array_diff_key(array_diff($old, $new), $new);
        $added = array_diff_key(array_diff($new, $old), $old);
        $changed = array_map(static function ($key) use ($old, $new) {
            return [
                'old' => $old[$key],
                'new' => $new[$key],
            ];
        }, array_keys(array_intersect_key(array_diff($new, $old), $old)));

        if ($deleted || $added || $changed) {
            $info = OrderHistory::getInstance()->getTrackingInfoLines(
                $added,
                $deleted,
                $changed
            );

            $i = 0;
            foreach ($info as $line) {
                $i++;
                static::setOrderChanges(
                    static::t('Order tracking information') . ":$i",
                    $line,
                    ''
                );
            }
        }
    }

    protected function sendOrderChangeNotification()
    {
        if (!$this->getOrder()->isNotificationSent()) {
            \XLite\Core\Mailer::sendOrderChanged(
                $this->getOrder(),
                $this->getIgnoreCustomerNotificationFlag()
            );
        }
    }

    /**
     * Get 'doNotSendNotification' flag from request
     *
     * @return boolean
     */
    protected function getIgnoreCustomerNotificationFlag()
    {
        return (bool) \XLite\Core\Request::getInstance()->doNotSendNotification;
    }

    /**
     * Update staff note
     */
    protected function updateAdminNotes()
    {
        $notes = \XLite\Core\Request::getInstance()->adminNotes;
        if (is_array($notes)) {
            $notes = reset($notes);
        }

        if (!$notes) {
            $notes = '';
        }

        $oldNotes = $this->getOrder()->getAdminNotes();

        if ($oldNotes != $notes) {
            $changes = [
                'old' => $oldNotes,
                'new' => $notes,
            ];

            \XLite\Core\OrderHistory::getInstance()
                ->registerOrderChangeAdminNotes($this->getOrder()->getOrderId(), $changes);

            $this->getOrder()->setAdminNotes($notes);

            \XLite\Core\Database::getEM()->flush();
        }
    }

    protected function updateCustomerNotes()
    {
        $notes = \XLite\Core\Request::getInstance()->notes;

        if (!is_null($notes)) {
            if (is_array($notes)) {
                $notes = reset($notes);
            }

            if (!$notes) {
                $notes = '';
            }

            $oldNotes = $this->getOrder()->getNotes();

            if ($oldNotes != $notes) {
                $changes = [
                    'old' => $this->getOrder()->getNotes(),
                    'new' => $notes,
                ];

                \XLite\Core\OrderHistory::getInstance()
                    ->registerOrderChangeCustomerNotes($this->getOrder()->getOrderId(), $changes);

                static::setOrderChanges(
                    static::t('Customer note'),
                    $changes['new'],
                    $changes['old']
                );

                $this->getOrder()->setNotes($notes);

                \XLite\Core\Database::getEM()->flush();
            }
        }
    }

    protected function updatePaymentMethods()
    {
        if ($this->isOrderEditable()) {
            $methods = \XLite\Core\Request::getInstance()->paymentMethods ?: [];

            foreach ($methods as $transactionId => $methodId) {
                $transaction = \XLite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->find($transactionId);
                $method = \XLite\Core\Database::getRepo('XLite\Model\Payment\Method')->find($methodId);
                $oldMethod = $transaction->getPaymentMethod();
                if ($method && $transaction && (!$oldMethod || $methodId != $oldMethod->getMethodId())) {
                    $transaction->setPaymentMethod($method);
                    \XLite\Core\OrderHistory::getInstance()
                        ->registerGlobalOrderChanges($this->getOrder()->getOrderId(), [
                            $this->getFieldHumanReadableName('paymentMethod') => [
                                'old'   => $oldMethod ? $oldMethod->getName() : null,
                                'new'   => $method->getName()
                            ]
                        ]);
                }
            }

            if ($method = $this->getOrder()->getPaymentMethod()) {
                $this->getOrder()->setPaymentMethod($method);
            }

            \XLite\Core\Database::getEM()->flush();
        }
    }

    /**
     * Send tracking information action
     */
    protected function doActionSendTracking()
    {
        \XLite\Core\Mailer::sendOrderTrackingInformationCustomer($this->getOrder());

        \XLite\Core\TopMessage::addInfo('Tracking information has been sent');
    }

    protected function doActionPaymentTransaction()
    {
        $order = $this->getOrder();

        if ($order) {
            $transactions = $order->getPaymentTransactions();
            if (!empty($transactions)) {
                foreach ($transactions as $transaction) {
                    if ($transaction->getTransactionId() == \XLite\Core\Request::getInstance()->trn_id) {
                        $transaction->getPaymentMethod()->getProcessor()->doTransaction(
                            $transaction,
                            \XLite\Core\Request::getInstance()->transactionType
                        );
                    }
                }
            }
        }
    }

    protected function doActionCalculatePrice()
    {
        if ($this->isOrderEditable()) {
            $request = \XLite\Core\Request::getInstance();

            $item = null;

            if ($request->item_id) {
                [$item, $attributes] = $this->getPreparedItemByItemId();
                if (!$item) {
                    $this->valid = false;
                }
            } elseif ($request->product_id) {
                [$item, $attributes] = $this->getPreparedItemByProductId();
            }

            if ($item) {
                $this->prepareItemBeforePriceCalculation($item, $attributes);
            }

            if (!$item) {
                $this->valid = false;
            } elseif (\XLite\Core\Request::getInstance()->isAJAX()) {
                $this->displayRecalculateItemPrice($item);
            }

            if (\XLite\Core\Request::getInstance()->isAJAX()) {
                $this->restoreFormId();
            }
        }
    }

    /**
     * Get prepared order item by item ID
     *
     * @return array
     */
    protected function getPreparedItemByItemId()
    {
        $order = $this->getOrder();
        $request = \XLite\Core\Request::getInstance();
        $attributeValues = [];
        $item = $order->getItemByItemId($request->item_id);

        if (
            $item
            && !empty($request->order_items[$request->item_id])
            && !empty($request->order_items[$request->item_id]['attribute_values'])
        ) {
            $attributeValues = $request->order_items[$request->item_id]['attribute_values'];
        }

        return [$item, $attributeValues];
    }

    /**
     * Get prepared order item by product ID
     *
     * @return array
     */
    protected function getPreparedItemByProductId()
    {
        $order = $this->getOrder();
        $request = \XLite\Core\Request::getInstance();
        $item = new \XLite\Model\OrderItem();
        $item->setOrder($order);
        $item->setProduct(\XLite\Core\Database::getRepo('XLite\Model\Product')->find($request->product_id));

        $attributes = $request->new;
        $attributes = !empty($attributes) ? reset($attributes) : null;
        $attributeValues = [];
        if (!empty($attributes['attribute_values'])) {
            $attributeValues = $attributes['attribute_values'];
        }

        return [$item, $attributeValues];
    }

    /**
     * Prepare order item before price calculation
     *
     * @param \XLite\Model\OrderItem $item       Order item
     * @param array                  $attributes Attributes
     */
    protected function prepareItemBeforePriceCalculation(\XLite\Model\OrderItem $item, array $attributes)
    {
        \XLite\Core\Request::getInstance()->oldAmount = $item->getAmount();

        $item->setAmount(\XLite\Core\Request::getInstance()->amount);

        if ($attributes) {
            $attributeValues = $item->getProduct()->prepareAttributeValues($attributes);
            $item->setAttributeValues($attributeValues);

            \XLite\Core\Database::getEM()->persist($item);
        }
    }

    /**
     * Display recalculate item price
     *
     * @param \XLite\Model\OrderItem $item Order item
     */
    protected function displayRecalculateItemPrice(\XLite\Model\OrderItem $item)
    {
        \XLite\Core\Event::recalculateItem($this->assembleRecalculateItemEvent($item));
    }

    /**
     * Assemble recalculate item event
     *
     * @param \XLite\Model\OrderItem $item Order item
     *
     * @return array
     */
    protected function assembleRecalculateItemEvent(\XLite\Model\OrderItem $item)
    {
        $maxAmount = $item->getProductAvailableAmount();

        if ($item->isPersistent() && \XLite\Core\Request::getInstance()->oldAmount) {
            $maxAmount += \XLite\Core\Request::getInstance()->oldAmount;
            \XLite\Core\Request::getInstance()->oldAmount = null;
        }

        return [
            'item_id'   => $item->getItemId(),
            'requestId' => \XLite\Core\Request::getInstance()->requestId,
            'price'     => $item->getNetPrice(),
            'max_qty'   => $maxAmount,
        ];
    }

    /**
     * getViewerTemplate
     *
     * @return string
     */
    protected function getViewerTemplate()
    {
        $result = parent::getViewerTemplate();

        if (\XLite\Core\Request::getInstance()->mode === 'invoice') {
            $result = 'common/print_invoice.twig';
        }

        if (\XLite\Core\Request::getInstance()->mode === 'packing_slip') {
            $result = 'common/print_packing_slip.twig';
        }

        return $result;
    }

    // {{{ Pages

    /**
     * Get pages sections
     *
     * @return array
     */
    public function getPages()
    {
        $list = parent::getPages();
        if (
            $this->getOrder()
            && Auth::getInstance()->isPermissionAllowed('ROLE_MANAGE_ORDERS')
        ) {
            $list['default'] = static::t('General info');
            $list['invoice'] = static::t('Invoice');
        }

        return $list;
    }

    /**
     * Get pages templates
     *
     * @return array
     */
    protected function getPageTemplates()
    {
        $list = parent::getPageTemplates();
        $list['default'] = 'order/page/info.tab.twig';
        $list['invoice'] = 'order/page/invoice.twig';

        return $list;
    }

    // }}}

    // {{{ Edit order

    /**
     * Get order form
     *
     * @return \XLite\View\Order\Details\Admin\Form
     */
    public function getOrderForm()
    {
        if (!isset($this->orderForm)) {
            $this->orderForm = new \XLite\View\Order\Details\Admin\Model();
        }

        return $this->orderForm;
    }

    /**
     * Service method: Get attributes for order status field widget.
     *
     * @param string $statusType Status type ('payment' or 'shipping')
     *
     * @return array
     */
    public function getOrderStatusAttributes($statusType)
    {
        return [
            'class' => 'not-affect-recalculate',
        ];
    }

    // }}}

    // {{{ Order surcharges

    /**
     * Get order surcharge totals
     *
     * @param boolean $force Force reassmeble modifiers OPTIONAL
     *
     * @return array
     */
    public function getSurchargeTotals($force = false)
    {
        if ($this->modifiers === null || $force) {
            $this->modifiers = $this->defineSurchargeTotals();
        }

        return $this->modifiers;
    }

    /**
     * Define surcharge totals
     *
     * @return array
     */
    protected function defineSurchargeTotals()
    {
        return $this->postprocessSurchargeTotals(
            $this->getOrderForm()->getModelObject()->getCompleteSurchargeTotals()
        );
    }

    /**
     * Postprocess surcharge totals
     *
     * @param array $modifiers Modifiers
     *
     * @return array
     */
    protected function postprocessSurchargeTotals(array $modifiers)
    {
        foreach ($modifiers as $code => $modifier) {
            $method = $this->assembleMethodNameByCodeModifier('postprocess%sSurcharge', $code);
            if (method_exists($this, $method)) {
                $modifiers[$code] = $this->$method($modifier);
            }
        }

        foreach ($this->getRequiredSurcharges() as $code) {
            if (!isset($modifiers[$code])) {
                $method = $this->assembleMethodNameByCodeModifier('assemble%sDumpSurcharge', $code);
                $dump = $this->$method();
                if ($dump && !isset($modifiers[$dump['code']])) {
                    $modifiers[$dump['code']] = $dump;
                }
            }
        }


        uasort($modifiers, static function ($a, $b) {
            $aWeight = $a['object']
                ? $a['object']->getSortingWeight()
                : 0;
            $bWeight = $b['object']
                ? $b['object']->getSortingWeight()
                : 0;
            return $aWeight < $bWeight ? -1 : $aWeight > $bWeight;
        });

        return $modifiers;
    }

    /**
     * Assemble method name by modifier's code
     *
     * @param string $pattern Pattern
     * @param string $code    Modifier's code
     *
     * @return string
     */
    protected function assembleMethodNameByCodeModifier($pattern, $code)
    {
        $code = preg_replace('/[^a-zA-Z0-9]/S', '', ucfirst(strtolower($code)));

        return sprintf($pattern, $code);
    }

    /**
     * Assemble shipping dump surcharge
     *
     * @return array
     */
    protected function assembleShippingDumpSurcharge()
    {
        return $this->assembleDefaultDumpSurcharge(
            \XLite\Model\Base\Surcharge::TYPE_SHIPPING,
            \XLite\Logic\Order\Modifier\Shipping::MODIFIER_CODE,
            '\XLite\Logic\Order\Modifier\Shipping',
            static::t('Shipping cost')
        );
    }

    /**
     * Assemble default dump surcharge
     *
     * @param string $type  Type
     * @param string $code  Code
     * @param string $class Class
     * @param string $name  Name
     *
     * @return array
     */
    protected function assembleDefaultDumpSurcharge($type, $code, $class, $name)
    {
        $surcharge = new \XLite\Model\Order\Surcharge();
        $surcharge->setType($type);
        $surcharge->setCode($code);
        $surcharge->setClass($class);
        $surcharge->setValue(0);
        $surcharge->setName($name);
        $surcharge->setOwner(static::getTemporaryOrder($this->getOrder()->getOrderId(), false) ?: $this->getOrder());

        return [
            'name'      => $surcharge->getTypeName(),
            'cost'      => $surcharge->getValue(),
            'available' => $surcharge->getAvailable(),
            'count'     => 1,
            'lastName'  => $surcharge->getName(),
            'code'      => $surcharge->getCode(),
            'widget'    => class_exists($class)
                ? $class::getWidgetClass()
                : \XLite\Logic\Order\Modifier\AModifier::getWidgetClass(),
            'object'    => $surcharge,
        ];
    }

    /**
     * Get required surcharges
     *
     * @return array
     */
    protected function getRequiredSurcharges()
    {
        return [
            \XLite\Logic\Order\Modifier\Shipping::MODIFIER_CODE,
        ];
    }

    // }}}
}
