<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Controller\Admin;

use XLite\Core\Request;
use XLite\Model\DTO\Base\ADTO;
use XLite\Model\DTO\Settings\Notification\Admin;
use XLite\Model\DTO\Settings\Notification\Customer;
use XLite\Model\Notification as NotificationModel;

class Notification extends \XLite\Controller\Admin\AAdmin
{
    use \XLite\Controller\Features\FormModelControllerTrait;

    public function __construct(array $params)
    {
        parent::__construct($params);

        $this->params = array_merge($this->params, ['page', 'templatesDirectory']);
    }

    public function handleRequest()
    {
        if ($this->getNotification()) {
            $this->handleNotificationPageParam();
        }

        parent::handleRequest();
    }

    /**
     * Redirect if url invalid
     */
    protected function handleNotificationPageParam()
    {
        $page = Request::getInstance()->page;

        if (!$this->isValidPage($page)) {
            if ($url = $this->buildValidPageUrl()) {
                $this->redirect($url);
            } else {
                $this->redirect($this->buildURL('notifications'));
            }
        }
    }

    protected function addBaseLocation()
    {
        parent::addBaseLocation();

        $this->addLocationNode(static::t('Email notifications'), $this->buildURL('notifications'));
    }

    /**
     * @param string $page
     *
     * @return bool
     */
    protected function isValidPage($page)
    {
        $notification = $this->getNotification();

        return in_array($page, [
                'admin',
                'customer',
            ], true)
            && (
                $page !== 'customer'
                || $notification?->getEnabledForCustomer()
                || $notification?->getAvailableForCustomer()
            )
            && (
                $page !== 'admin'
                || $notification?->getEnabledForAdmin()
                || $notification?->getAvailableForAdmin()
            );
    }

    /**
     * @return string
     */
    protected function buildValidPageUrl()
    {
        $notification = $this->getNotification();

        if ($notification) {
            if ($notification->getAvailableForCustomer() || $notification->getEnabledForCustomer()) {
                return $this->buildURL('notification', '', [
                    'templatesDirectory' => $notification->getTemplatesDirectory(),
                    'page'               => 'customer',
                ]);
            }

            if ($notification->getAvailableForAdmin() || $notification->getEnabledForAdmin()) {
                return $this->buildURL('notification', '', [
                    'templatesDirectory' => $notification->getTemplatesDirectory(),
                    'page'               => 'admin',
                ]);
            }
        }

        return '';
    }

    /**
     * @return string
     */
    public function getTitle()
    {
        $notification = $this->getNotification();

        return $notification
            ? $notification->getName()
            : '';
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        $notification = $this->getNotification();

        return $notification
            ? $notification->getDescription()
            : '';
    }

    /**
     * Returns object to get initial data and populate submitted data to
     *
     * @return ADTO
     */
    public function getFormModelObject()
    {
        $class = $this->getFormModelObjectClass();

        return new $class($this->getNotification());
    }

    /**
     * @return string
     */
    protected function getFormModelObjectClass()
    {
        return Request::getInstance()->page === 'admin'
            ? Admin::class
            : Customer::class;
    }

    protected function doActionUpdate()
    {
        $dto = $this->getFormModelObject();
        $formModel = new \XLite\View\FormModel\Settings\Notification\Notification(['object' => $dto]);

        $form = $formModel->getForm();
        $data = \XLite\Core\Request::getInstance()->getData();
        $rawData = \XLite\Core\Request::getInstance()->getNonFilteredData();

        $form->submit($data[$this->formName]);

        if ($form->isValid()) {
            $dto->populateTo($this->getNotification(), $rawData[$this->formName]);
            \XLite\Core\Database::getEM()->flush();

            \XLite\Core\TopMessage::addInfo('The notification has been updated');
        } else {
            $this->saveFormModelTmpData($rawData[$this->formName]);
        }
    }

    /**
     * @return NotificationModel|null
     */
    public function getNotification()
    {
        $templatesDirectory = \XLite\Core\Request::getInstance()->templatesDirectory;

        return $templatesDirectory
            ? \XLite\Core\Database::getRepo(NotificationModel::class)
                ?->findOneBy(['templatesDirectory' => $templatesDirectory, 'available' => 1])
            : null;
    }

    /**
     * @return bool
     */
    protected function isVisible()
    {
        return parent::isVisible() && $this->getNotification();
    }

    /**
     * @return string
     */
    public function getPage()
    {
        return $this->page;
    }
}
