<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Console\Command\Utils;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use XLite\Core\Config;
use XLite\Logic\Import\Console\Importer;

class Import extends Command
{
    public function __construct(
        private Filesystem $filesystem,
        private EntityManagerInterface $entityManager
    ) {
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setName('utils:import')
            ->setDescription('Import data from a csv file')
            ->setHelp('')
            ->addArgument('file', InputArgument::REQUIRED, 'File to import')
            ->addOption(
                'use-workers',
                'w',
                InputOption::VALUE_OPTIONAL,
                '(experimental) Use workers (minimum 3 recommended) during import step for performance gains'
            )
            ->addOption(
                'skip-verification',
                '',
                InputOption::VALUE_NONE,
                '(For advanced users) Skip the verification step. Use this when you are certain the data is accurate and straightforward, such as importing qty/price only'
            )
            ->addOption(
                'trusted-data',
                't',
                InputOption::VALUE_OPTIONAL,
                'Skip HTML purification of string values'
            )
            ->addOption(
                'delimiter',
                'd',
                InputOption::VALUE_OPTIONAL,
                'Сsv file separator'
            )
            ->addOption(
                'remove-file',
                'r',
                InputOption::VALUE_NONE,
                'Remove import files'
            )
        ;
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        // TODO: really bad to provide $io to importer in such a direct manner - too fragile.
        // Need a wrapper for this $io and an interface.
        $io = new SymfonyStyle($input, $output);

        if (
            $this->consoleImportIsStarted()
            && $io->ask('Console import is started. Continue anyway?[no/yes]', 'yes') !== 'yes'
        ) {
            return Command::FAILURE;
        }

        $file = $input->getArgument('file');
        $delim = $input->getOption('delimiter') ?? Config::getInstance()->Units->csv_delim ?? ',';
        $needRemove = $input->getOption('remove-file');

        $options = [
            'useWorkers'       => (bool)$input->getOption('use-workers'),
            'skipVerification' => (bool)$input->getOption('skip-verification'),
            'trustedData'      => (bool)$input->getOption('trusted-data'),
            'delimiter'        => $delim,
        ];

        if (!file_exists($file)) {
            throw new \RuntimeException("File $file not exists.");
        }

        if (is_dir($file)) {
            $finder = Finder::create()->files()->in($file)->name(['*.csv', '*.jsonl']);
            if (!$finder->count()) {
                throw new \RuntimeException("Directory $file is empty.");
            }

            $files = $this->sortingFiles(iterator_to_array($finder));
        } else {
            $files = [$file];
        }

        $this->defineStartImportFlag();

        try {
            foreach ($files as $file) {
                $io->info("Import: $file");

                $options['file'] = $file;
                $importer = new Importer($options, $io);
                $importer->run();

                if ($needRemove) {
                    $this->filesystem->remove($file);
                }
            }
        } finally {
            $this->removeImportFlag();
        }

        return Command::SUCCESS;
    }

    /**
     * @param SplFileInfo[] $files File paths
     */
    protected function sortingFiles(array $files): array
    {
        usort(
            $files,
            fn (SplFileInfo $first, SplFileInfo $second) => $this->getFileIndex($first->getFilename()) <=> $this->getFileIndex($second->getFilename())
        );

        return $files;
    }

    protected function getFileIndex(string $file): int
    {
        $filePrefixes = [
            'categories', 'attributes', 'products', 'product-attributes', 'users', 'orders'
        ];

        foreach ($filePrefixes as $key => $filePrefix) {
            if (str_starts_with($file, $filePrefix)) {
                return $key;
            }
        }

        return count($filePrefixes);
    }

    protected function consoleImportIsStarted(): bool
    {
        return (bool) $this->entityManager->getRepository(\XLite\Model\TmpVar::class)->getVar('Console import start');
    }

    protected function defineStartImportFlag(): void
    {
        $this->entityManager->getRepository(\XLite\Model\TmpVar::class)->setVar('Console import start', true);
    }

    protected function removeImportFlag(): void
    {
        $this->entityManager->getRepository(\XLite\Model\TmpVar::class)->removeVar('Console import start');
    }
}
