<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Console\Command\GenerateData\Generators;

use XC\ProductVariants\Model\ProductVariant;
use XCart\Domain\ModuleManagerDomain;
use XLite\Console\Command\GenerateData\GlobalRuntimeCache;
use XLite\Core\Database;
use XLite\Model\Attribute;
use XLite\Model\AttributeValue\AttributeValueSelect;

/**
 * Class Product
 * @package XLite\Console\Command\GenerateData\Generators
 */
class Product
{
    private $options;
    private $optionsValues;
    private $productImages;
    private $wholesalePrices;
    /**
     * @var Image
     */
    private $imagesGenerator;

    private ModuleManagerDomain $moduleManagerDomain;

    public function __construct(
        $options,
        $optionsValues,
        $productImages,
        $wholesalePrices,
        $imagesGenerator
    ) {
        $this->options = $options;
        $this->optionsValues = $optionsValues;
        $this->productImages = $productImages;
        $this->wholesalePrices = $wholesalePrices;
        $this->imagesGenerator = $imagesGenerator;
        $this->moduleManagerDomain = \XCart\Container::getContainer()->get(ModuleManagerDomain::class);
    }

    /**
     * @param $category
     * @param $suffix
     *
     * @return \XLite\Model\Product
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     */
    public function generate($category, $suffix)
    {
        $product = $this->generateProductItself($category, $suffix);

        for ($i = 0; $i < $this->productImages; $i++) {
            $this->generateImages($product);
        }

        if ($this->moduleManagerDomain->isEnabled('CDev-Wholesale')) {
            $this->generateWholesalePrices($product, $this->wholesalePrices);
        }

        $res = $this->generateGlobalAttributeValues($product);

        for ($i = 0; $i < $this->options; $i++) {
            $this->generateOptions($product, $i, $this->optionsValues);
        }

        if (
            $this->moduleManagerDomain->isEnabled('XC-ProductVariants')
            && !empty($res)
        ) {
            $this->generateVariants($product, $res);
        }

        return $product;
    }

    public function generateGlobalAttributeValues(\XLite\Model\Product $product): array
    {
        $result = [];
        foreach (GlobalRuntimeCache::getAttributes() as $a) {
            foreach ($a->getAttributeOptions() as $ao) {
                $av = new AttributeValueSelect();
                $av->setProduct($product);
                $av->setAttribute($a);
                $av->setAttributeOption($ao);
                $av->setPosition($ao->getPosition());
                Database::getEM()->persist($av);
                if (Database::getEM()->contains($av)) {
                    $result[] = $av;
                }
            }
        }
        return $result;
    }

    protected function generateProductItself(\XLite\Model\Category $category, $suffix)
    {
        /** @var \XLite\Model\Product $product */
        $product = Database::getRepo('XLite\Model\Product')->insert(
            [
                'sku'    => 'SKU' . $category->getCategoryId() . '_' . $suffix,
                'name'   => 'Test product #' . $category->getCategoryId() . ' - ' . $suffix,
                'price'  => mt_rand(1, 100),
                'weight' => mt_rand(1, 100),
            ],
            false
        );
        $link = new \XLite\Model\CategoryProducts();
        $link->setProduct($product);
        $link->setCategory($category);
        $product->addCategoryProducts($link);
        Database::getEM()->persist($link);

        return $product;
    }

    protected function generateImages(\XLite\Model\Product $product)
    {
        $image = new \XLite\Model\Image\Product\Image();
        $image->setProduct($product);

        if ($image->loadFromLocalFile($this->imagesGenerator->generateImage())) {
            $product->addImages($image);
        }
    }

    /**
     * @param \XLite\Model\Product $product
     * @param                      $count
     */
    protected function generateWholesalePrices(\XLite\Model\Product $product, $count)
    {
        $q1 = 1;
        $q2 = 10;
        for ($i = 0; $i < $count; $i++) {
            $last = $i === $count - 1;
            Database::getRepo('CDev\Wholesale\Model\WholesalePrice')->insert(
                [
                    'quantityRangeBegin' => $q1,
                    'quantityRangeEnd'   => $last ? 0 : $q2,
                    'product'            => $product,
                    'price'              => $product->getPrice() * round(mt_rand(0, 100) / 100, 2),
                ],
                false
            );
            $q1 = $q2 + 1;
            $q2 += 10;
        }

        Database::getRepo('CDev\Wholesale\Model\MinQuantity')->insert(
            [
                'product'  => $product,
                'quantity' => 1,
            ],
            false
        );
    }

    /**
     * @param \XLite\Model\Product $product
     * @param                      $suffix
     * @param                      $optionsValuesCount
     *
     * @throws \Doctrine\ORM\ORMInvalidArgumentException
     */
    protected function generateOptions(\XLite\Model\Product $product, $suffix, $optionsValuesCount)
    {
        $attribute = Database::getRepo(Attribute::class)->insert(
            [
                'product'     => $product,
                'name'        => 'Test option ' . $product->getProductId() . '-' . $suffix,
                'displayMode' => Attribute::TYPE_SELECT
            ],
            false
        );
        Database::getEM()->persist($attribute);
        for ($n = 0; $n < $optionsValuesCount; $n++) {
            $option = Database::getRepo('XLite\Model\AttributeOption')->insert(
                [
                    'attribute' => $attribute,
                    'name'      => 'value ' . $suffix . '-' . $n,
                ],
                false
            );
            Database::getEM()->persist($option);
            $value = Database::getRepo('XLite\Model\AttributeValue\AttributeValueSelect')->insert(
                [
                    'attribute'        => $attribute,
                    'product'          => $product,
                    'attribute_option' => $option,
                    'priceModifier'    => $n > 0 ? round(mt_rand(0, 50) / 10, 1) : 0,
                ],
                false
            );
            Database::getEM()->persist($value);
        }
    }

    protected function generateVariants(\XLite\Model\Product $product, array $attributeValues): void
    {
        $this->updateVariantsAttributes($product);

        $variants = [];
        $variantsAttributes = $this->getVariantsAttributes($product);

        /** @var Attribute $a */
        foreach ($variantsAttributes as $a) {
            $_variants = $variants;
            $variants = [];
            /** @var AttributeValueSelect $attributeValue */
            foreach ($attributeValues as $attributeValue) {
                if ($attributeValue->getAttribute()->getId() !== $a->getId()) {
                    continue;
                }
                $val = [[$attributeValue, $a->getType()]];
                if ($_variants) {
                    foreach ($_variants as $v) {
                        $variants[] = array_merge($val, $v);
                    }
                } else {
                    $variants[] = $val;
                }
            }
        }

        if ($variants) {
            foreach ($variants as $attributeValues) {
                $variant = new ProductVariant();
                foreach ($attributeValues as $attributeValue) {
                    $method = 'addAttributeValue' . $attributeValue[1];
                    $attributeValue = $attributeValue[0];
                    $variant->$method($attributeValue);
                    $attributeValue->addVariants($variant);
                }
                $variant->setVariantId(
                    md5(uniqid(rand(), true))
                );
                $variant->setProduct($product);
                $product->addVariants($variant);
                Database::getEM()->persist($variant);
            }
        }
    }

    public function getVariantsAttributes(\XLite\Model\Product $product): array
    {
        return $product->getVariantsAttributes()->toArray();
    }

    protected function updateVariantsAttributes(\XLite\Model\Product $product)
    {
        foreach (GlobalRuntimeCache::getAttributes() as $attribute) {
            $product->addVariantsAttributes($attribute);
        }
    }
}
