<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\API;

use ApiPlatform\Core\OpenApi\Factory\OpenApiFactoryInterface;
use ApiPlatform\Core\OpenApi\Model\MediaType;
use ApiPlatform\Core\OpenApi\Model\Parameter;
use ApiPlatform\Core\OpenApi\Model\RequestBody;
use ApiPlatform\Core\OpenApi\Model\Response;
use ApiPlatform\Core\OpenApi\OpenApi;
use ApiPlatform\OpenApi\Model\Operation;
use ApiPlatform\OpenApi\Model\PathItem;
use ArrayObject;

class OpenApiFactory implements OpenApiFactoryInterface
{
    protected OpenApiFactoryInterface $decorated;

    protected OpenApiTagsRepositoryInterface $tagsRepository;

    public function __construct(OpenApiFactoryInterface $decorated, OpenApiTagsRepositoryInterface $tagsRepository)
    {
        $this->decorated      = $decorated;
        $this->tagsRepository = $tagsRepository;
    }

    public function __invoke(array $context = []): OpenApi
    {
        $openApi = $this->decorated->__invoke($context);

        // x-tagGroups
        $tagGroups = [];
        foreach ($this->tagsRepository->getTags() as $group => $tags) {
            $tagGroup = [
                'name' => $group,
                'tags' => [],
            ];
            foreach ($tags as $tag) {
                $tagGroup['tags'][] = $tag;
            }
            $tagGroups[] = $tagGroup;
        }
        $openApi = $openApi->withExtensionProperty('tagGroups', $tagGroups);

        // custom enriching
        $pathItem  = $openApi->getPaths()->getPath('/api/categories/{category_id}/products');
        $operation = $pathItem->getGet();
        $openApi->getPaths()->addPath('/api/categories/{category_id}/products', $pathItem->withGet(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() === 'category_id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/products/{product_id}/images/{image_id}');
        $operation = $pathItem->getGet();
        $openApi->getPaths()->addPath('/api/products/{product_id}/images/{image_id}', $pathItem->withGet(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/products/{product_id}/images/{image_id}');
        $operation = $pathItem->getDelete();
        $openApi->getPaths()->addPath('/api/products/{product_id}/images/{image_id}', $pathItem->withDelete(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/products/{product_id}/images/{image_id}');
        $operation = $pathItem->getPut();
        $openApi->getPaths()->addPath('/api/products/{product_id}/images/{image_id}', $pathItem->withPut(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/categories/{category_id}/icon');
        $operation = $pathItem->getGet();
        $openApi->getPaths()->addPath('/api/categories/{category_id}/icon', $pathItem->withGet(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/categories/{category_id}/icon');
        $operation = $pathItem->getDelete();
        $openApi->getPaths()->addPath('/api/categories/{category_id}/icon', $pathItem->withDelete(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/categories/{category_id}/banner');
        $operation = $pathItem->getGet();
        $openApi->getPaths()->addPath('/api/categories/{category_id}/banner', $pathItem->withGet(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));

        $pathItem  = $openApi->getPaths()->getPath('/api/categories/{category_id}/banner');
        $operation = $pathItem->getDelete();
        $openApi->getPaths()->addPath('/api/categories/{category_id}/banner', $pathItem->withDelete(
            $operation->withParameters(
                array_filter($operation->getParameters(), static fn (Parameter $p) => $p->getName() !== 'id'),
            ),
        ));


        // Storefront API custom enriching

        // api/storefront/carts/{cart_id}/calculatedShippingRates
        $pathItem = $openApi->getPaths()->getPath('/api/storefront/carts/{cart_id}/calculatedShippingRates');
        $postOperation = $pathItem->getPost();
        $postResponses = $postOperation->getResponses();
        unset($postResponses['422']);

        $openApi->getPaths()->addPath(
            '/api/storefront/carts/{cart_id}/calculatedShippingRates',
            $pathItem->withPost($postOperation->withResponses($postResponses))
        );

        // api/storefront/carts/{cart_id}/items/{id}
        $pathItem = $openApi->getPaths()->getPath('/api/storefront/carts/{cart_id}/items/{id}');
        $patchOperation = $pathItem->getPatch();
        $patchResponses = $patchOperation->getResponses();
        unset($patchResponses['422']);

        $openApi->getPaths()->addPath(
            '/api/storefront/carts/{cart_id}/items/{id}',
            $pathItem->withPatch($patchOperation->withResponses($patchResponses))
        );

        // api/storefront/carts/{cart_id}/order
        $pathItem = $openApi->getPaths()->getPath('/api/storefront/carts/{cart_id}/order');
        $postOperation = $pathItem->getPost();
        $postResponses = $postOperation->getResponses();
        unset($postResponses['422']);

        $openApi->getPaths()->addPath(
            '/api/storefront/carts/{cart_id}/order',
            $pathItem->withPost($postOperation->withResponses($postResponses))
        );

        // api/storefront/users/self
        $pathItem = $openApi->getPaths()->getPath('/api/storefront/users/self');

        $getOperation = $pathItem->getGet();
        $getResponses = $getOperation->getResponses();
        unset($getResponses['404']);

        $patchOperation = $pathItem->getPatch();
        $patchResponses = $patchOperation->getResponses();
        unset($patchResponses['404']);

        $deleteOperation = $pathItem->getDelete();
        $deleteResponses = $deleteOperation->getResponses();
        unset($deleteResponses['404']);

        $openApi->getPaths()->addPath(
            '/api/storefront/users/self',
            $pathItem
                ->withGet($getOperation->withResponses($getResponses))
                ->withPatch($patchOperation->withResponses($patchResponses))
                ->withDelete($deleteOperation->withResponses($deleteResponses))
        );


        // api/storefront/auth/login
        $openApi->getComponents()->getSchemas()['Auth.Token'] = new ArrayObject(
            [
                'type'       => 'object',
                'properties' => [
                    'token'                    => [
                        'type' => 'string',
                    ],
                    'refresh_token'            => [
                        'type'      => 'string',
                        'minLength' => 128,
                        'maxLength' => 128,
                    ],
                    'refresh_token_expiration' => [
                        'type'    => 'integer',
                        'minimum' => 1,
                    ],
                ],
            ]
        );

        $responses = [];

        $responses['200'] = new Response(
            'User authorized',
            new ArrayObject(
                [
                    'application/json'         => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                    'application/ld+json'      => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                    'application/vnd.api+json' => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                    'text/html'                => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                ]
            ),
        );

        $responses['400'] = new Response('Invalid input');
        $responses['401'] = new Response('Invalid credentials');

        $requestBody = new RequestBody(
            'User credentials',
            new ArrayObject(
                [
                    'application/json' => new MediaType(
                        new ArrayObject(
                            [
                                'type'       => 'object',
                                'properties' => [
                                    'username' => [
                                        'type' => 'string',
                                    ],
                                    'password' => [
                                        'type' => 'string',
                                    ],
                                ],
                            ]
                        )
                    ),
                ]
            ),
        );

        $pathItem = (new PathItem())
            ->withPost(
                new Operation(
                    'apiStorefrontAuthLogin',
                    ['Authentication'],
                    $responses,
                    'Login',
                    '',
                    null,
                    [],
                    $requestBody,
                ),
            );
        $openApi->getPaths()->addPath('/api/storefront/auth/login', $pathItem);

        // api/storefront/auth/refresh
        $responses = [];

        $responses['200'] = new Response(
            'User authorized',
            new ArrayObject(
                [
                    'application/json'         => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                    'application/ld+json'      => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                    'application/vnd.api+json' => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                    'text/html'                => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Auth.Token'])
                    ),
                ]
            ),
        );
        $responses['401'] = new Response('JWT Refresh Token Not Found | Missing JWT Refresh Token | Invalid JWT Refresh Token');

        $requestBody = new RequestBody(
            'Refresh token',
            new ArrayObject(
                [
                    'application/json' => new MediaType(
                        new ArrayObject(
                            [
                                'type'       => 'object',
                                'properties' => [
                                    'refresh_token' => [
                                        'type' => 'string',
                                    ],
                                ],
                            ]
                        )
                    ),
                ]
            ),
        );

        $pathItem = (new PathItem())
            ->withPost(
                new Operation(
                    'apiStorefrontAuthRefresh',
                    ['Authentication'],
                    $responses,
                    'Refresh token',
                    '',
                    null,
                    [],
                    $requestBody,
                ),
            );
        $openApi->getPaths()->addPath('/api/storefront/auth/refresh', $pathItem);

        // api/storefront/auth/logout
        $responses = [];

        $logoutResponseMediaType = new MediaType(
            new ArrayObject(
                [
                    'type'       => 'object',
                    'properties' => [
                        'code'    => [
                            'type'    => 'integer',
                            'example' => 200,
                        ],
                        'message' => [
                            'type'    => 'string',
                            'example' => 'The supplied refresh_token has been invalidated.',
                        ],
                    ],
                ]
            )
        );

        $logoutResponseContent = new ArrayObject(
            [
                'application/json'         => $logoutResponseMediaType,
                'application/ld+json'      => $logoutResponseMediaType,
                'application/vnd.api+json' => $logoutResponseMediaType,
                'text/html'                => $logoutResponseMediaType,
            ]
        );

        $responses['200'] = new Response(
            'The supplied refresh_token has been invalidated',
            $logoutResponseContent
        );
        $responses['400'] = new Response(
            'No refresh_token found',
            $logoutResponseContent
        );
        $responses['401'] = new Response(
            'Expired JWT Token',
            $logoutResponseContent
        );

        $requestBody = new RequestBody(
            'Refresh token',
            new ArrayObject(
                [
                    'application/json' => new MediaType(
                        new ArrayObject(
                            [
                                'type'       => 'object',
                                'properties' => [
                                    'refresh_token' => [
                                        'type' => 'string',
                                    ],
                                ],
                            ]
                        )
                    ),
                ]
            ),
        );

        $pathItem = (new PathItem())
            ->withPost(
                new Operation(
                    'apiStorefrontAuthLogout',
                    ['Authentication'],
                    $responses,
                    'Logout',
                    '',
                    null,
                    [],
                    $requestBody,
                ),
            );
        $openApi->getPaths()->addPath('/api/storefront/auth/logout', $pathItem);

        return $openApi;
    }
}
