/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Items list controller
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

/**
 * Items list controller
 */
function TableItemsList(cell, URLParams, URLAJAXParams)
{
  TableItemsList.superclass.constructor.apply(this, arguments);
  xcart.bind('reflow-tables', _.bind(this.onReflowTables, this));
}

extend(TableItemsList, ItemsList);

TableItemsList.prototype.form = null;

TableItemsList.prototype.newLinesIndex = 0;

TableItemsList.prototype.binded = false;

TableItemsList.prototype.mainHeaderHeight = 40;

TableItemsList.prototype.floatHeaderZIndex = 10;

// Set a param and send the request
TableItemsList.prototype.process = function(paramName, paramValue)
{
  var form = this.container.parents('form').get(0);

  var result = true;

  if (!form || !form.commonController.isChanged(true)) {
    result = ItemsList.prototype.process.apply(this, arguments);
  }

  return result;
};

// Place new list content
TableItemsList.prototype.placeNewContent = function(content)
{
  TableItemsList.superclass.placeNewContent.apply(this, arguments);
  var form = this.container.parents('form').get(0);

  if (form && form.commonController) {
    form.commonController.bindElements();
    jQuery(form).trigger('change');
  }

  xcart.trigger('stickyPanelReposition');
};

// Pager listener
TableItemsList.prototype.listeners.pager = function(handler)
{
  jQuery('.table-pager .input input', handler.container).change(
    function() {
      return !handler.process('pageId', this.value - 1);
    }
  );

  jQuery('.table-pager a', handler.container).click(
    function() {
      jQuery('html, body').animate({scrollTop: handler.container.offset().top});

      return !(jQuery(this).hasClass('disabled') || handler.process('pageId', jQuery(this).data('pageid')));
    }
  );

};

// Item per page input listener
TableItemsList.prototype.listeners.pagesCount = function(handler)
{
  jQuery('select.page-length', handler.container).change(
    function() {
      return !handler.process('itemsPerPage', this.options[this.selectedIndex].value);
    }
  );
};

// Form listener
TableItemsList.prototype.listeners.form = function(handler)
{
  var form = handler.container.parents('form').eq(0);

  if (form.get(0) && form.get(0).commonController) {
    form.get(0).commonController.submitOnlyChanged = false;
  }

  if (!TableItemsList.prototype.binded) {
    TableItemsList.prototype.binded = true;
    form.bind('submit', function (event) {
      if (form.hasClass('confirm-remove') && form.find('.line.remove-mark').length > 0) {
        if (!confirm(xcart.t('Do you really want to delete selected items?'))) {
          event.preventDefault();
          event.stopImmediatePropagation();
        }
      }
    });
  }

  form.bind('submit.prevalidate', function () {
    handler.updateChangedMarks(jQuery(this));
  });

  form.bind(
    'state-changed',
    function () {
      if (typeof processFormChanged !== 'undefined') {
        handler.processFormChanged(jQuery(this));
      }
    }
  );
  form.bind(
      'more-action-enabled',
      function () {
        if (typeof processFormChanged !== 'undefined') {
          handler.processFormChanged(jQuery(this));
        }
      }
  );
  form.bind(
    'state-initial',
    function () {
      if (typeof processFormChanged !== 'undefined') {
        handler.processFormChanged(jQuery(this));
      }
    }
  );
  form.bind(
    'more-action-initial',
    function () {
      if (typeof processFormChanged !== 'undefined') {
        handler.processFormChanged(jQuery(this));
      }
    }
  );

};

TableItemsList.prototype.updateChangedMarks = function(form)
{
  this.clearChangedMarks(form);
  _.each(this.getChangedIds(form.get(0).getElements()), _.bind(function (id) {
    this.addChangedMark(form, id);
  }, this));
};

// Process form and form's elements after form changed
TableItemsList.prototype.processFormChanged = function(form)
{
  this.container.find('.table-pager .input input, .table-pager .page-length').each(
    function () {
      jQuery(this).prop('disabled', 'disabled');
      this.setAttribute('disabled', 'disabled');
    }
  );

  this.markPagerAsDisabled();
};

TableItemsList.prototype.getChangedIds = function(elements)
{
  return _.map(
    _.filter(elements, function (i) { return i.name && i.name.search(/^data\[\d+]/) !== -1 && i.isChanged(); }),
    function (i) { return i.name.replace(/data\[(\d+)].*/, '$1'); }
  );
};

TableItemsList.prototype.addChangedMark = function(form, id)
{
  jQuery('<input type="hidden" name="data[' + id + '][_changed]" value="true" class="changed-mark" />').appendTo(form);
};

TableItemsList.prototype.clearChangedMarks = function(form)
{
  form.find('.changed-mark').remove();
};

TableItemsList.prototype.markPagerAsDisabled = function()
{
  var pagerBlock = this.container.find('.table-pager');

  if (pagerBlock.length > 0) {
    assignShadeOverlay(pagerBlock).css('z-index', 3);
    pagerBlock.find('a').addClass('disabled').removeClass('enabled');
  };
}

TableItemsList.prototype.markPagerAsEnabled = function()
{
  var pagerBlock = this.container.find('.table-pager');

  if (pagerBlock.length > 0) {
    unassignShadeOverlay(pagerBlock);
    pagerBlock.find('a').removeClass('disabled').addClass('enabled');
  };
}


// Process form and form's elements after form cancel all changes
TableItemsList.prototype.processFormUndo = function(form)
{
  this.container.find('.table-pager .input input, .table-pager .page-length').removeAttr('disabled');
  this.markPagerAsEnabled();
};

// Inline creation button listener
TableItemsList.prototype.listeners.createButton = function(handler)
{
  jQuery('tbody.create :input', handler.container)
    .addClass('no-validate');

  jQuery('button.create-inline', handler.container)
    .prop('onclick', null)
    .click(
      function (event) {

        event.stopPropagation();

        var box = jQuery('tbody.create', handler.container);
        var length = box.find('.line').length;
        handler.newLinesIndex++;
        var idx = handler.newLinesIndex;
        var line = box.find('.create-tpl').clone(true);
        line
          .show()
          .removeClass('create-tpl')
          .addClass('create-line')
          .addClass('line')
          .find(':input')
          .unbind()
          .removeClass('no-validate')
          .each(
            function () {
              if (this.id) {
                this.id = this.id.replace(/-0-/, '-n' + idx + '-');
              }
              this.name = this.name.replace(/\[0\]/, '[' + (-1 * idx) + ']');
              jQuery(this).data('init-value', 'false');
            }
          )
          .end()
          .find('label')
          .each(
            function () {
              if (this.attributes['for']) {
                this.attributes['for'].value = this.attributes['for'].value.replace(/-0-/, '-n' + idx + '-');
              }
            }
          );

        line.find('.invalid-inline-field').removeClass('invalid-inline-field');
        box.append(line);
        _.defer(function(){
          jQuery(':input', line).eq(0).focus();
        });

        var form = box.parents('form').get(0);
        if (form) {
          form.commonController.bindElements();
        }

        jQuery('table.list', handler.container).removeClass('list-no-items');
        jQuery('.table-wrapper', handler.container).removeClass('empty');
        jQuery('.no-items', handler.container).hide();
        jQuery('.sticky-panel').css('display', 'block').height(jQuery('.sticky-panel').find('.box').eq(0).height());
        jQuery('.additional-panel').removeClass('hidden').show();
        if (handler.container.length) {
          handler.container[0].parentNode.querySelectorAll('.list-is-empty').forEach(
            function(item) {
              item.classList.remove('list-is-empty');
            }
          )
        }

        if (2 == box.children('tr').length) {
          var rightAction = jQuery('tbody.lines tr td.actions.right', handler.container).eq(0);
          if (rightAction.length) {
            line.find('td.actions.right').width(rightAction.width())
          }
        }

        line.parents('.items-list').get(0).itemsListController.triggerVent('newLineCreated', {idx: idx, line: line});

        xcart.trigger('itemListNewItemCreated', {idx: idx, line: line});

        handler.triggerVent('line.new.add');

        xcart.trigger('reflow-tables', true);

        if (this.form) {
          jQuery(this.form).change();
        }

        return false;
      }
    );
};

// Selector actions
TableItemsList.prototype.listeners.selector = function(handler)
{
  jQuery('.actions div.selector', handler.container).click(
    function () {
      var input = jQuery('input', this).get(0);
      if (input) {
        input.checked = !input.checked;
        jQuery('input', this).change();
      }
    }
  );

  jQuery('.actions input.selector', handler.container).click(
    function (event) {
      event.stopPropagation();
    }
  );

  jQuery('.actions input.selector', handler.container).click(
    function (event) {
      event.stopPropagation();

      const $box = jQuery('.actions', handler.container);
      const $selectAll = jQuery('input.selectAll', handler.container);
      const $unselectAll = jQuery('input.unselectAll', handler.container);

      const checkedInputsExist = $box.find('input.selector:checked').length;

      if (checkedInputsExist) {
        $selectAll.prop('checked', false).hide();
        $unselectAll.prop('checked', true).show();
      } else {
        $selectAll.prop('checked', false).show();
        $unselectAll.prop('checked', false).hide();
      }
    }
  );

  jQuery('.actions input.selector', handler.container).change(
    _.bind(
      function (event) {
        const box = jQuery(event.target).parent('div.selector');
        const oldSelected = 0 < jQuery('.actions div.selector.checked', handler.container).length;

        const row = jQuery(event.target).closest('tr');

        if (event.target.checked) {
          box.addClass('checked');
          row.addClass('selected-row');
        } else {
          box.removeClass('checked');
          row.removeClass('selected-row');
        }

        var newSelected = 0 < jQuery('.actions div.selector.checked', handler.container).length;

        if (oldSelected != newSelected) {
          if (newSelected) {
            this.triggerVent('selector.checked', {widget: handler});

          } else {
            this.triggerVent('selector.unchecked', {widget: handler});
          }
        }

        return true;
      },
      handler
    )
  );

  jQuery('.actions div.default-value-checkbox input', handler.container).click(
    function (event) {
      jQuery('.actions div.default-value-checkbox', handler.container).removeClass('checked');
      jQuery(this).parent().parent().addClass('checked');
      event.stopPropagation();
    }
  );

  jQuery('.actions div.default-value-checkbox', handler.container).click(
    function () {
      jQuery(this).find('input[type=radio]').click();
    }
  );

  jQuery('input.selectAll', handler.container).click(
    function () {
      var box = jQuery('.actions', handler.container);
      var checked = jQuery(this).prop('checked');
      var last = null;
      box.find('input.selector').each(
        function (index, elem) {
          if (checked != jQuery(elem).prop('checked')) {
            jQuery(elem).prop('checked', checked);
            jQuery(elem).closest('tr').addClass('selected-row');
            last = elem;
          }
        }
      );

      if (last) {
        jQuery(last).prop('checked', !checked);
        jQuery(last).click();
      }

      if (checked) {
        box.find('div.selector', handler.container).addClass('checked');
        handler.triggerVent('selector.massChecked', { widget: handler, 'checked': checked });

      } else {
        box.find('div.selector', handler.container).removeClass('checked');
        handler.triggerVent('selector.massUnchecked', { widget: handler, 'checked': checked });
      }
    }
  );

  jQuery('input.unselectAll', handler.container).click(
    function () {
      const box = jQuery('.actions', handler.container);
      const $selectAll = jQuery('input.selectAll', handler.container);
      const $unselectAll = jQuery('input.unselectAll', handler.container);
      box.find('input.selector').each(
        function (index, elem) {
          if (jQuery(elem).prop('checked') === true) {
            jQuery(elem).prop('checked', false);
            jQuery(elem).closest('tr').removeClass('selected-row');
          }
        }
      );
      box.find('div.selector', handler.container).removeClass('checked');
      $unselectAll.hide();
      $selectAll.prop('checked', false).show();
    }
  );

};

// Position changed
TableItemsList.prototype.listeners.positionChanged = function(handler)
{
  jQuery('tbody.lines', handler.container).bind(
    'positionChange',
    function () {
      var i = 0;
      var length = jQuery(this).find('.lines').length;
      jQuery(this).find('.lines').each(
        function () {
          var tr = jQuery(this);

          if (0 == i) {
            tr.addClass('first');
          } else {
            tr.removeClass('first');
          }

          if (length - 1 == i) {
            tr.addClass('last');
          } else {
            tr.removeClass('last');
          }

          if (0 == (i + 1) % 2) {
            tr.addClass('even');
          } else {
            tr.removeClass('even');
          }
        }
      );
    }
  );
};

// Head sort
TableItemsList.prototype.listeners.headSort = function(handler)
{
  jQuery('thead th a.sort', handler.container).click(
    function() {
      if (jQuery(this).hasClass('current-sort')
        && jQuery(this).hasClass('single-order-sort')
      ) {
        return false;
      }

      return jQuery(this).hasClass('current-sort')
        ? !handler.process('sortOrder', 'asc' == jQuery(this).data('direction') ? 'desc' : 'asc')
        : !handler.process('sortBy', jQuery(this).data('sort'));
    }
  );
};

// Head search
TableItemsList.prototype.listeners.headSearch = function(handler)
{
  jQuery('tbody.head-search input,tbody.head-search select', handler.container).change(
    function() {
      var result = false;
      jQuery(this).parents('td').eq(0).find('input,select,textarea').each(
        function () {
          result = handler.setURLParam(this.name, this.value) || result;
        }
      );

      if (result) {
        handler.loadWidget();
      }

      return false;
    }
  );
};

// Fade resize
TableItemsList.prototype.listeners.fadeResize = function(handler)
{
  /*
  var func = function(isResize)
  {
    var cells = jQuery('tbody td.no-wrap .cell', handler.container);
    if (isResize) {

      // Reset width
      cells.each(
        function() {
          jQuery(this).width(jQuery(this).data('initial-width'));
        }
      );
    }

    cells.each(
      function() {

        // Save initial width
        if (!jQuery(this).data('initial-width')) {
          jQuery(this).data('initial-width', jQuery(this).width());
        }

        // Set new width
        var td = jQuery(this).parents('td').eq(0).width();
        if (td > jQuery(this).outerWidth()) {
          jQuery(this).width(td);
        }
      }
    );
  }

  jQuery(window).resize(_.bind(func, this, true));
  func();
  */
};

// Reassign items list controller
TableItemsList.prototype.reassign = function()
{
  new TableItemsList(this.params.cell, this.params.urlparams, this.params.urlajaxparams);
};

// Get event namespace (prefix)
TableItemsList.prototype.getEventNamespace = function()
{
  return 'list.model.table';
}

TableItemsList.prototype.initialize = function(elem, urlparams, urlajaxparams)
{
  const result = ItemsList.prototype.initialize.apply(this, arguments);
  this.addFloatThead();
  this.addHoverEventHandler();

  return result;
};

TableItemsList.prototype.addFloatThead = function ()
{
  if (this.isFloatTheadAllowed()) {
    const $table = jQuery('table.list');
    $table.on('floatThead', _.bind(this.onFloatThead, this));
    const options = {
      top: this.mainHeaderHeight,
      zIndex: this.floatHeaderZIndex,
      position: 'absolute'
    }
    $table.floatThead(options);

    const $tableImages = $table.find('img');

    if ($tableImages.length) {
      const reflowFloatThead = () => {
        $table.floatThead('reflow');
      };

      window.addEventListener('load', reflowFloatThead);
      $table.on('floatThead', reflowFloatThead);

      $tableImages.each(
        function () {
          this.addEventListener('load', reflowFloatThead);
          this.addEventListener('error', reflowFloatThead);
        }
      );

      reflowFloatThead();
    }
  }
}

TableItemsList.prototype.onFloatThead = function (e, isFloated, $floatContainer)
{
  if (isFloated) {
    $floatContainer.addClass('floated');
  } else {
    $floatContainer.removeClass('floated');
  }
}

TableItemsList.prototype.isFloatTheadAllowed = function ()
{
  const isOrderItems = jQuery('.order-items').length;

  return !isOrderItems;
}

TableItemsList.prototype.addHoverEventHandler = function () {
  const $td = jQuery('table.list td').has('div.editable');
  $td.hover(function () {
      const thisIndex = jQuery(this).parents('tr').find('td').index(jQuery(this));
      jQuery('table tr td:nth-child(' + (thisIndex + 1) + ')').addClass('hover');
    },
    function () {
      $td.removeClass('hover');
    });
}

TableItemsList.prototype.onReflowTables = function (event, immediately = false) {
  setTimeout(() => {
    let $table = jQuery('table.list');
    $table.css('min-width', '');
    $table.trigger('reflow');
  }, immediately ? 0 : 500);
}

function TableItemsListQueue()
{
  jQuery('.widget.items-list').each(
    function (index, elem)
    {
      elem = jQuery(elem);
      var cls = elem.data('js-class');
      cls = (cls && typeof(window[cls]) != 'undefined')
        ? window[cls]
        : TableItemsList;

      new cls(elem);
    }
  );
}

xcart.autoload(TableItemsListQueue);

jQuery().ready(
  function() {
    //jQuery('.items-list').width(jQuery('.items-list').width());

    xcart.microhandlers.add(
      'ItemsListWidth',
      '.table-pager',
      function (event) {
        jQuery('.items-list').each(function (index, elem) {
          // jQuery('.list', elem).width(jQuery('.table-pager', elem).width() - 2);
        });
      }
    );

  }
);
